import logging

import cars.settings
from cars.carsharing.models.car import Car
from cars.core.util import import_class
from cars.orders.core.order_manager import OrderManager
from cars.orders.models.order_item import OrderItem
from .push_client import PUSH_CLIENT


LOGGER = logging.getLogger(__name__)


class CarCommander(object):

    class NotAuthorized(Exception):
        pass

    class NotFound(Exception):
        pass

    def __init__(self, telematics_proxy, order_manager):
        self._telematics_proxy = telematics_proxy
        self._order_manager = order_manager

    @classmethod
    def from_settings(cls):
        telematics_proxy_class = import_class(cars.settings.TELEMATICS['proxy_class'])
        telematics_proxy = telematics_proxy_class.from_settings()
        order_manager = OrderManager.from_settings(push_client=PUSH_CLIENT)
        obj = cls(
            telematics_proxy=telematics_proxy,
            order_manager=order_manager,
        )
        return obj

    def blink(self, user, car):
        self._check_car_command_is_authorized(user, car)
        response = self._telematics_proxy.blink(car.imei)
        self._check_command_response(response)

    def open(self, user, car):
        self._check_car_command_is_authorized(user, car)
        response = self._telematics_proxy.open(car.imei)
        self._check_command_response(response)

    def close(self, user, car):
        self._check_car_command_is_authorized(user, car)
        response = self._telematics_proxy.close(car.imei)
        self._check_command_response(response)

    def start_lease(self, user, car):
        self._check_car_command_is_authorized(user, car)
        response = self._telematics_proxy.start_lease(car.imei)
        self._check_command_response(response)

    def end_lease(self, user, car):
        self._check_car_command_is_authorized(user, car)
        response = self._telematics_proxy.end_lease(car.imei)
        self._check_command_response(response)

    def warmup(self, user, car):
        self._check_car_command_is_authorized(user, car)
        response = self._telematics_proxy.warmup(car.imei)
        self._check_command_response(response)

    def _check_car_command_is_authorized(self, user, car):
        authorized_statuses = {
            Car.Status.ACCEPTANCE,
            Car.Status.RESERVATION,
            Car.Status.RESERVATION_PAID,
            Car.Status.PARKING,
            Car.Status.RIDE,
        }
        if car.get_status() not in authorized_statuses:
            LOGGER.warning(
                'unauthorized car command: car status %s user=%s car=%s',
                car.get_status().name,
                user.id,
                car.id,
            )
            raise self.NotAuthorized

        order_item = self._order_manager.get_active_order_item(
            user=user,
            order_item_types=[
                OrderItem.Type.CARSHARING_ACCEPTANCE,
                OrderItem.Type.CARSHARING_PARKING,
                OrderItem.Type.CARSHARING_RESERVATION,
                OrderItem.Type.CARSHARING_RESERVATION_PAID,
                OrderItem.Type.CARSHARING_RIDE,
            ],
            with_related=True,
        )
        if order_item is None:
            LOGGER.warning(
                'unauthorized car command: missing order user=%s car=%s',
                user.id,
                car.id,
            )
            raise self.NotAuthorized

        if order_item.get_impl().car_id != car.id:
            LOGGER.error(
                'unauthorized car command: ordered other car %s user=%s car=%s',
                order_item.get_impl().car_id,
                user.id,
                car.id,
            )
            raise self.NotAuthorized

    def _check_command_response(self, response):
        response.raise_for_status()
