import collections
import datetime
import itertools
import json
import logging
import re

from cars.settings import INCIDENT

from django.db import transaction

from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.core.startrek_client import StartrekClient
from cars.core.util import collection_to_mapping, datetime_helper

from cars.drive.serializers.incident import UserInfoSerializer, SessionInfoSerializer
from cars.request_aggregator.models.call_center_common import CallStatSyncStatus, SyncOrigin

from . import template as template_holder

from .service import handle_evacuation_error, EvacuationException
from .template import DESCRIPTION_TEMPLATE, SUMMARY_TEMPLATE, DATA_CHANGE_TEMPLATE
from .template_matching import TemplateMatcher


LOGGER = logging.getLogger(__name__)


TagInfo = collections.namedtuple('TagInfo', ('name', 'priority'))


class EvacuationManagementHelper(object):
    POSSIBLE_EVACUATION_TAG_INFO = TagInfo('possible_evacuation', 1000)
    EVACUATION_TAG_INFO = TagInfo('evacuation', 1000)

    USER_TICKET_TAG_NAME = 'user_ticket_st'
    USER_CHARACTER_TAG_NAME = 'user_character_eva'

    RAW_EVA_TIME_TEMPLATES = (
        '%Y-%m-%d %H:%M:%S', '%d-%m-%Y %H:%M:%S', '%d.%m.%Y %H:%M:%S', '%Y.%m.%d %H:%M:%S',
        '%Y-%m-%d %H:%M', '%d-%m-%Y %H:%M', '%d.%m.%Y %H:%M', '%Y.%m.%d %H:%M',
        '%H:%M:%S %Y-%m-%d', '%H:%M:%S %d-%m-%Y', '%H:%M:%S %d.%m.%Y', '%H:%M:%S %Y.%m.%d',
    )

    def __init__(self, saas_client, st_client, st_ticket_queue_mapping, description_template_name_mapping):
        self._saas_client = saas_client
        self._st_client = st_client

        self._st_ticket_queue_mapping = st_ticket_queue_mapping

        self._description_template_mapping = {
            k: DESCRIPTION_TEMPLATE + getattr(template_holder, v)
            for k, v in description_template_name_mapping.items()
        }

        ticket_queues = '|'.join(v['queue'] for v in st_ticket_queue_mapping.values())
        self._st_ticket_key_re = re.compile('((?:{})-\d+)'.format(ticket_queues))

        self._template_matcher_collection = [TemplateMatcher(t) for t in self._description_template_mapping.values()]

    @classmethod
    def from_settings(cls):
        saas_client = SaasDriveAdminClient.from_settings()
        st_client = StartrekClient.from_settings()
        st_ticket_queue_mapping = INCIDENT['evacuation']['ticket_queue']
        description_template_name_mapping = INCIDENT['evacuation']['template_name']
        return cls(saas_client, st_client, st_ticket_queue_mapping, description_template_name_mapping)

    def add_evacuation_info(self, data, request):
        response_data = {'message': None}

        car_data, user_data, session_data, evacuation_data = self._extract_data(data)

        do_process_car_tags = True
        do_process_user_tags = not user_data['is_default']  # session data may be absent (default)

        car_tags = self._get_car_tags(car_data)

        self._check_multiple_possible_evacuation_tags(car_tags)

        existing_issue_key = self._get_related_car_issue_key(car_tags)

        if existing_issue_key is not None:
            self._check_need_assign_user_ticket(existing_issue_key, user_data, do_process_user_tags)

            ticket_info = self._st_client.get_issue(existing_issue_key)
            do_process_car_tags = False
        else:
            performer = request.user
            ticket_info = self._make_st_ticket(performer, car_data, session_data, user_data, evacuation_data)

        with self._saas_client.with_custom_auth(request=request):
            if do_process_car_tags:
                self._process_car_tags(ticket_info, car_tags, car_data, response_data)

            if do_process_user_tags:
                self._add_user_tags(ticket_info, car_data, session_data, user_data)

        return response_data

    def _check_multiple_possible_evacuation_tags(self, car_tags):
        possible_evacuation_tags = car_tags.get(self.POSSIBLE_EVACUATION_TAG_INFO.name, [None])

        if len(possible_evacuation_tags) > 1:
            raise EvacuationException(detail='car has more than 2 active possible evacuation tags')

    def _get_related_car_issue_key(self, car_tags):
        car_evacuation_tags = car_tags.get(self.EVACUATION_TAG_INFO.name, [None])

        if len(car_evacuation_tags) > 1:
            raise EvacuationException(detail='car has more than 2 active evacuation tags')

        (car_evacuation_tag, ) = car_evacuation_tags

        existing_issue_key = None

        if car_evacuation_tag is not None:
            car_related_tickets = self._st_ticket_key_re.findall(car_evacuation_tag.get('comment', ''))

            if len(car_related_tickets) == 1:
                (existing_issue_key, ) = car_related_tickets
            elif len(car_related_tickets) > 1:
                raise EvacuationException(
                    detail='more than 1 ticket is assigned to car: {}'.format(car_related_tickets)
                )
            else:
                raise EvacuationException(
                    detail='car has evacuation tags but no related tickets found'
                )

        return existing_issue_key

    def _check_need_assign_user_ticket(self, existing_issue_key, user_data, do_process_user_tags):
        if do_process_user_tags:
            user_tags = self._get_user_tags(user_data)
            user_evacuation_tags = user_tags.get(self.USER_TICKET_TAG_NAME, [])

            user_related_tickets = set(
                itertools.chain.from_iterable(
                    self._get_related_ticket_keys(tag) for tag in user_evacuation_tags
                )
            )

            if existing_issue_key in user_related_tickets:
                raise EvacuationException(
                    detail='a ticket {} has been already assigned to both car and user'.format(existing_issue_key)
                )
        else:
            raise EvacuationException(detail=(
                'a ticket {} has been already assigned to car '
                'and not required to be assigned to user'.format(existing_issue_key)
            ))

    def update_evacuation_info_from_tag(self, user_id, tag_id):
        user_id, tag_id = str(user_id), str(tag_id)

        user_tags = self._saas_client.get_user_tags(user_id, re_raise=True)
        specific_tag = next((tag for tag in user_tags if tag['tag_id'] == tag_id), None)

        if specific_tag is None:
            LOGGER.warning('tag {} specified is not still active for user {}'.format(tag_id, user_id))
            return

        st_ticket_keys = self._get_related_ticket_keys(specific_tag)

        if not st_ticket_keys:
            LOGGER.info('no evacuation ticket tags found for user {}'.format(user_id))
            return
        elif len(st_ticket_keys) > 1:
            raise Exception('multiple ticket urls found for user {} in tag {}'.format(user_id, tag_id))
        else:
            (st_ticket_key,) = st_ticket_keys

        user_data = self._serialize_data({'id': user_id}, UserInfoSerializer)

        session_id = specific_tag.get('session_id', None) or None
        session_data = self._serialize_data({'id': session_id}, SessionInfoSerializer)

        self._modify_evacuation_info(st_ticket_key, user_data, session_data)

    def _get_related_ticket_keys(self, specific_tag):
        st_ticket_keys = list(
            itertools.chain.from_iterable(
                self._st_ticket_key_re.findall(l.get('uri', '')) for l in specific_tag.get('links', [])
                if l.get('type', None) == 'st'
            )
        )
        return st_ticket_keys

    def _serialize_data(self, data, serializer_cls):
        data_serializer = serializer_cls(data=data)
        data_serializer.is_valid(raise_exception=True)
        validated_data = data_serializer.validated_data
        return validated_data

    def _modify_evacuation_info(self, issue_key, user_data_update, session_data_update):
        issue_info = self._st_client.get_issue(issue_key)

        original_description = issue_info['description']

        matcher = None
        original_data = None
        is_partial = False

        for matcher in self._template_matcher_collection:
            try:
                original_data, is_partial = matcher.process_text(original_description, allow_partial=False)
            except Exception:
                pass

        if original_data is None:
            original_data, is_partial = matcher.process_text(original_description, allow_partial=True)

        try:
            car_data, user_data, session_data, evacuation_data = self._extract_data(original_data)
            original_user_id, new_user_id = user_data['id'], user_data_update['id']
            original_session_id, new_session_id = session_data['id'], session_data_update['id']
        except (AttributeError, KeyError):
            raise EvacuationException(
                detail='cannot match ticket description correctly: issue - {}'.format(issue_key)
            )

        if original_user_id == new_user_id and original_session_id == new_session_id:
            LOGGER.info(
                'no evacuation data changed for user {} and session {}'
                .format(original_user_id, original_session_id)
            )
        elif is_partial:
            raise EvacuationException(detail=(
                'cannot match ticket description correctly: '
                'issue - {}, extracted user id - {}, extracted session id - {}'
                .format(issue_key, original_user_id, original_session_id)
            ))
        else:
            self._modify_updated_user_data(issue_key, original_data, session_data_update, user_data_update)

    def _modify_updated_user_data(self, issue_key, original_data, session_data_update, user_data_update):
        car_data, user_data, session_data, evacuation_data = self._extract_data(original_data)

        original_user_id, new_user_id = user_data['id'], user_data_update['id']
        original_session_id, new_session_id = session_data['id'], session_data_update['id']

        LOGGER.info(
            'modifying evacuation data for user {} and session {}'
            .format(original_user_id, original_session_id)
        )

        user_data.update(user_data_update)
        session_data.update(session_data_update)

        summary = self._format_summary(car_data, user_data)
        template = self._get_description_template(evacuation_data)
        description = self._format_description(template, car_data, session_data, user_data, evacuation_data)

        self._st_client.patch_issue(issue_key, summary=summary, description=description)

        issue_update_comment = self._format_data_change(
            original_user_id=original_user_id, new_user_id=new_user_id,
            original_session_id=original_session_id, new_session_id=new_session_id
        )
        self._st_client.add_issue_comment(issue_key, text=issue_update_comment)

        LOGGER.info(
            'evacuation data changed successfully from user {} to {} and from session {} to {}'
            .format(original_user_id, new_user_id, original_session_id, new_session_id)
        )

    def _extract_data(self, data):
        return [data.get(x, None) for x in ('car', 'user', 'session', 'evacuation')]

    @handle_evacuation_error('startrek')
    def _make_st_ticket(self, performer, car_data, session_data, user_data, evacuation_data):
        queue_info = self._get_ticket_queue_info(evacuation_data)
        queue, generic_extra_fields = queue_info['queue'], queue_info.get('extra', {})

        summary = self._format_summary(car_data, user_data)

        template = self._get_description_template(evacuation_data)
        description = self._format_description(template, car_data, session_data, user_data, evacuation_data)

        followers = [performer.username]

        extra_fields = {'stateNumber': car_data['number']}

        if evacuation_data.get('timestamp'):
            localized_evacuation_date = evacuation_data.get('timestamp')
            formatted_evacuation_date = localized_evacuation_date.strftime('%Y-%m-%dT%H:%M:%S.000%z')
            extra_fields['evacuationDate'] = formatted_evacuation_date

        elif evacuation_data.get('datetime'):
            # awaiting frontend to be removed
            raw_evacuation_date = evacuation_data.get('datetime')

            for time_template in self.RAW_EVA_TIME_TEMPLATES:
                try:
                    parsed_evacuation_date = datetime.datetime.strptime(raw_evacuation_date, time_template)
                except ValueError:
                    pass
                else:
                    localized_evacuation_date = datetime_helper.localize(parsed_evacuation_date)
                    formatted_evacuation_date = localized_evacuation_date.strftime('%Y-%m-%dT%H:%M:%S.000%z')
                    extra_fields['evacuationDate'] = formatted_evacuation_date
                    break
            else:
                LOGGER.error('cannot interpret evacuation date as a datetime object: {}'.format(raw_evacuation_date))

        if evacuation_data.get('address'):
            extra_fields['violationPlace'] = evacuation_data.get('address')

        if generic_extra_fields:
            extra_fields.update(generic_extra_fields)

        issue_info = self._st_client.create_issue(
            queue=queue, summary=summary, description=description, followers=followers, **extra_fields
        )
        return issue_info

    def _get_ticket_queue_info(self, evacuation_data):
        return self._st_ticket_queue_mapping[evacuation_data['city']]

    def _get_description_template(self, evacuation_data):
        return self._description_template_mapping[evacuation_data['city']]

    def _format_summary(self, car_data, user_data):
        return SUMMARY_TEMPLATE.format(car=car_data, user=user_data)

    def _format_description(self, template, car_data, session_data, user_data, evacuation_data):
        return template.format(
            car=car_data, session=session_data, user=user_data, evacuation=evacuation_data
        )

    def _format_data_change(self, **kwargs):
        return DATA_CHANGE_TEMPLATE.format(**kwargs)

    def _process_car_tags(self, ticket_info, car_tags, car_data, response_data):
        possible_evacuation_tags = car_tags.get(self.POSSIBLE_EVACUATION_TAG_INFO.name, [None])

        (possible_evacuation_tag, ) = possible_evacuation_tags

        if possible_evacuation_tag is not None:
            tag_data_to_evolve = self._prepare_tag_data_to_evolve(possible_evacuation_tag, ticket_info)
            self._schedule_car_tag_evaluation(tag_data_to_evolve)
            LOGGER.info('tag evaluation is scheduled: id - {}'.format(tag_data_to_evolve['tag_id']))
            response_data['message'] = 'tag evaluation is scheduled successfully'

        else:
            if car_data['add_tag']:
                self._add_car_tags(ticket_info, car_data)

    def _get_car_tags(self, car_data):
        current_car_tags = self._saas_client.get_car_tags(car_data['id'], re_raise=True)
        grouped_tags = collection_to_mapping(current_car_tags, item_key='tag')
        return grouped_tags

    def _get_user_tags(self, user_data):
        current_user_tags = self._saas_client.get_user_tags(user_data['id'], re_raise=True)
        grouped_tags = collection_to_mapping(current_user_tags, item_key='tag')
        return grouped_tags

    def _prepare_tag_data_to_evolve(self, current_tag, ticket_info):
        tag_id = str(current_tag['tag_id'])

        comment = self._get_ticket_url(ticket_info)
        if 'comment' in current_tag:
            comment += '\n' + current_tag['comment']

        tag_update_data = {
            'tag_id': tag_id,
            'comment': comment,
            'tag_name': self.EVACUATION_TAG_INFO.name,
            'priority': self.EVACUATION_TAG_INFO.priority,
        }

        return tag_update_data

    @handle_evacuation_error('add_car_tag')
    def _add_car_tags(self, ticket_info, car_data):
        self._saas_client.add_car_tag(
            car_id=car_data['id'],
            tag_name=self.EVACUATION_TAG_INFO.name,
            comment=self._get_ticket_url(ticket_info),
            priority=self.EVACUATION_TAG_INFO.priority,
            re_raise=True,
        )

    @handle_evacuation_error('add_user_tag')
    def _add_user_tags(self, ticket_info, car_data, session_data, user_data):
        session_id = session_data['id'] if not session_data['is_default'] else ''

        self._saas_client.add_user_problem_tag(
            user_id=user_data['id'],
            tag=self.USER_TICKET_TAG_NAME,
            car_number=car_data['number'],
            session_id=session_id,
            st_links=self._get_ticket_url(ticket_info),
            re_raise=True,
        )

        self._saas_client.add_user_tag(
            user_id=user_data['id'],
            tag=self.USER_CHARACTER_TAG_NAME,
            re_raise=True,
        )

    @transaction.atomic(savepoint=False)
    def _schedule_car_tag_evaluation(self, tag_data_to_evolve):
        sync_entry = (
            CallStatSyncStatus.objects.select_for_update()
            .get(origin=SyncOrigin.EVACUATION_TAG_EVOLUTION_MONITORING.value)
        )

        if sync_entry.data is None:
            sync_entry.data = {}

        sync_entry.data.setdefault('tags_to_evolve', [])
        sync_entry.data['tags_to_evolve'].append(tag_data_to_evolve)

        sync_entry.save()

    def _get_ticket_url(self, ticket_info):
        return '{}/{}'.format(self._st_client.web_url, ticket_info['key'])


class EvacuationTagsHelper(object):
    def __init__(self, saas_client, default_performer):
        self._saas_client = saas_client
        self._default_performer = default_performer

    @classmethod
    def from_settings(cls):
        saas_client = SaasDriveAdminClient.from_settings(timeout=10)
        default_performer = INCIDENT['default_performer_id']
        return cls(saas_client, default_performer)

    @handle_evacuation_error('evolve_car_tag')
    def add_tag(self, car_id, tag, priority, comment, *, check_exist=True, exclude=()):
        car_tags = self._get_car_tags(car_id)

        if any((exclude_tag in car_tags) for exclude_tag in exclude):
            return

        if not check_exist or tag not in car_tags:
            self._saas_client.add_car_tag(car_id, tag, comment, priority, re_raise=True)

    def _get_car_tags(self, car_id, group_by='tag', multiple_values=True):
        current_car_tags = self._saas_client.get_car_tags(car_id, re_raise=True)
        grouped_tags = collection_to_mapping(current_car_tags, item_key=group_by, multiple_values=multiple_values)
        return grouped_tags

    @handle_evacuation_error('evolve_car_tag')
    def remove_tag(self, car_id, tag, *, tag_filter=None):
        car_tags = self._get_car_tags(car_id)
        specific_tags = car_tags.get(tag, [])

        for specific_tag in specific_tags:
            if tag_filter is not None and not tag_filter(specific_tag):
                continue

            specific_tag_id = specific_tag['tag_id']
            self._saas_client.remove_car_tag(specific_tag_id, re_raise=True)

    @handle_evacuation_error('evolve_car_tag')
    def evolve_tag(self, tag_data_to_evolve, tag_instance_mapping):
        car_id = tag_instance_mapping['object_id']
        tag_id = tag_data_to_evolve['tag_id']

        extra_robot_token = INCIDENT['evolution_extra_robot_token']
        extra_performer = INCIDENT['extra_performer_id']

        if tag_instance_mapping['tag'] != tag_data_to_evolve['tag_name']:
            if tag_instance_mapping.get('performer', None) not in (self._default_performer, extra_performer):
                with self._saas_client.with_custom_auth(token=extra_robot_token):
                    self._saas_client.set_car_tag_performer(tag_id, re_raise=True)

            with self._saas_client.with_custom_auth(token=extra_robot_token):
                self._saas_client.evolve_car_tag(re_raise=True, **tag_data_to_evolve)

        current_tag_state = self._get_car_tags(car_id, group_by='tag_id', multiple_values=False).get(tag_id, None)

        if current_tag_state is not None and current_tag_state.get('performer', None) in (self._default_performer, extra_performer):
            with self._saas_client.with_custom_auth(token=extra_robot_token):
                self._saas_client.drop_car_tag_performer(tag_id_to_drop=tag_id, re_raise=True)
