import collections
import contextlib
import datetime
import json
import logging

import cars.settings

from cars.core.solomon import SolomonHelper
from cars.core.history import HistoryManager
from cars.core.util import datetime_helper

from cars.request_aggregator.core.request_time_sync_helper import RequestTimeSyncHelper
from cars.request_aggregator.models.call_center_common import SyncOrigin
from cars.users.models import UserTagHistory

from cars.carsharing.models.tag import CarTag
from cars.cars.models.car_tag import CarTagHistory

from .management_helper import EvacuationManagementHelper, EvacuationTagsHelper
from .service import EvacuationException


LOGGER = logging.getLogger(__name__)


TagInfo = collections.namedtuple('TagInfo', ('name', 'priority'))


class EvacuationTicketMonitoringHelper(object):
    manager = EvacuationManagementHelper.from_settings()
    solomon_helper = SolomonHelper('drive', 'evacuation_tag_monitor')

    def __init__(self):
        self._sync_helper = RequestTimeSyncHelper(
            SyncOrigin.EVACUATION_USER_TAG_MONITORING,
            default_since=datetime_helper.utc_now(),
        )

    @classmethod
    def from_settings(cls):
        return cls()

    def process(self):
        for since, until in self._sync_helper.iter_time_span_to_process():
            self._update_evacuation_info(since, until)

    def _update_evacuation_info(self, since, until):
        user_ticket_st_tags_history = (
            UserTagHistory.objects
            .filter(
                history_timestamp__gte=since,
                history_timestamp__lt=until,
                tag='user_ticket_st',
                history_action__in=(
                    HistoryManager.HistoryAction.ADD.value,
                    HistoryManager.HistoryAction.UPDATE_DATA.value,
                ),
            )
            .values('object_id', 'tag_id')
        )

        counters = collections.Counter()

        for tag_history_entry in user_ticket_st_tags_history:
            user_id = str(tag_history_entry['object_id'])
            tag_id = str(tag_history_entry['tag_id'])

            try:
                self.manager.update_evacuation_info_from_tag(user_id=user_id, tag_id=tag_id)
            except EvacuationException:
                counters['fail'] += 1
                LOGGER.exception('error processing tag entry: tag id - {}, user id - {}'.format(tag_id, user_id))
                self._sync_helper.add_extra_data('st_update_error_tag_id', tag_id)
            else:
                counters['success'] += 1

        self.solomon_helper.increment_counters(**counters)


class EvacuationSTSTagMonitoringHelper(object):
    manager = EvacuationTagsHelper.from_settings()
    solomon_helper = SolomonHelper('drive', 'evacuation_sts_tag_monitor')

    STS_TAG_INFO = TagInfo('get_STS', 1100)
    POSSIBLE_EVACUATION_TAG_INFO = EvacuationManagementHelper.POSSIBLE_EVACUATION_TAG_INFO
    EVACUATION_TAG_INFO = EvacuationManagementHelper.EVACUATION_TAG_INFO

    def __init__(self):
        self._sync_helper = RequestTimeSyncHelper(
            SyncOrigin.EVACUATION_STS_TAG_MONITORING,
            default_since=datetime_helper.utc_now(),
        )

        self._evacuation_tags_to_monitor = (
            self.POSSIBLE_EVACUATION_TAG_INFO.name,
            self.EVACUATION_TAG_INFO.name,
        )

        self._tags_to_monitor = (self.STS_TAG_INFO.name, ) + self._evacuation_tags_to_monitor

    @classmethod
    def from_settings(cls):
        return cls()

    def process(self):
        for since, until in self._sync_helper.iter_time_span_to_process():
            self._update_sts_tags(since, until)

    def _update_sts_tags(self, since, until):
        car_tags_history = (
            CarTagHistory.objects
            .using(cars.settings.DB_RO_ID)
            .filter(
                history_timestamp__gte=since,
                history_timestamp__lt=until,
                tag__in=self._tags_to_monitor,
            )
            .values('object_id', 'tag', 'tag_id', 'history_action')
        )

        counters = collections.Counter()

        for tag_history_entry in car_tags_history:
            history_action = tag_history_entry['history_action']
            tag_id = str(tag_history_entry['tag_id'])

            if (
                    history_action == HistoryManager.HistoryAction.ADD.value and
                    not CarTag.objects.filter(tag_id=tag_id).exists()
            ):
                LOGGER.info('skipping tag {} add entry as tag was removed'.format(tag_id))
                counters['skip'] += 1
                continue

            try:
                self._update_tag(tag_history_entry)
            except Exception:
                LOGGER.exception('error processing tag {} {} entry'.format(tag_id, history_action))
                counters['fail'] += 1
            else:
                LOGGER.info('tag {} {} entry has been successfully processed'.format(tag_id, history_action))
                counters['success'] += 1

        self.solomon_helper.increment_counters(**counters)

    def _update_tag(self, tag_history_entry):
        car_id = str(tag_history_entry['object_id'])
        history_action = tag_history_entry['history_action']

        tag = str(tag_history_entry['tag'])
        tag_id = str(tag_history_entry['tag_id'])

        if tag in self._evacuation_tags_to_monitor:
            if history_action == HistoryManager.HistoryAction.ADD.value:
                if (
                        tag == self.EVACUATION_TAG_INFO.name and
                        CarTag.objects.filter(object_id=car_id, tag=self.POSSIBLE_EVACUATION_TAG_INFO.name).exists()
                ):
                    LOGGER.info(
                        'do not add sts tag for car {} after adding tag {} {}'
                        ' as it has been processed adding possible evacuation one'.format(car_id, tag, tag_id)
                    )
                else:
                    LOGGER.info(
                        'adding sts tag for car {} after adding tag {} {}'
                        .format(car_id, tag, tag_id)
                    )
                    comment = 'automatically derived from {}'.format(tag_id)
                    self.manager.add_tag(
                        car_id, self.STS_TAG_INFO.name, self.STS_TAG_INFO.priority,
                        comment, check_exist=True
                    )

            elif history_action == HistoryManager.HistoryAction.REMOVE.value:
                if not CarTag.objects.filter(object_id=car_id, tag=tag).exists():
                    LOGGER.info(
                        'removing possible eva and sts tags for car {} after removing tag {} {}'
                        .format(car_id, tag, tag_id)
                    )
                    self.manager.remove_tag(car_id, self.POSSIBLE_EVACUATION_TAG_INFO.name)
                    self.manager.remove_tag(car_id, self.STS_TAG_INFO.name)
                else:
                    LOGGER.info(
                        'do not remove possible eva and sts tags for car {} after removing tag {} {}'
                        ' as another tag {} exists'.format(car_id, tag, tag_id, tag)
                    )

            else:
                pass

        elif tag == self.STS_TAG_INFO.name:
            if history_action == HistoryManager.HistoryAction.ADD.value:
                comment = 'automatically derived from {}'.format(tag_id)
                self.manager.add_tag(
                    car_id, self.POSSIBLE_EVACUATION_TAG_INFO.name, self.POSSIBLE_EVACUATION_TAG_INFO.priority,
                    comment, check_exist=False, exclude=self._evacuation_tags_to_monitor
                )

        else:
            raise RuntimeError('improper tag name')


class EvacuationTagEvolutionMonitoringHelper(object):
    manager = EvacuationTagsHelper.from_settings()
    solomon_helper = SolomonHelper('drive', 'evacuation_sts_tag_monitor')

    TELEMATIC_PROBLEMS_TAG_INFO = TagInfo('telematic_problems', 1000)

    def __init__(self):
        self._sync_helper = RequestTimeSyncHelper(
            SyncOrigin.EVACUATION_TAG_EVOLUTION_MONITORING,
            default_since=datetime_helper.utc_now(),
        )

    @classmethod
    def from_settings(cls):
        return cls()

    def process_tags_to_evolve(self):
        if not self._sync_helper.check_is_active():
            return

        tags_to_evolve = list(self._sync_helper.get_extra_data('tags_to_evolve', []))

        counters = collections.Counter()

        for tag_data in tags_to_evolve:
            remove_tag_data, sensor_suffix = True, 'success'

            tag_id = tag_data['tag_id']
            tag_instance_mapping = (
                CarTag.objects.filter(tag_id=tag_id)
                .values('tag_id', 'tag', 'object_id', 'performer')
                .first()
            )

            if tag_instance_mapping is not None:
                try:
                    self.manager.evolve_tag(tag_data, tag_instance_mapping)

                except EvacuationException as exc:
                    LOGGER.error('error evolving scheduled tag {}: {}'.format(tag_id, json.dumps(exc.data)))

                    if exc.error_code == 409:
                        if exc.error_code_name == 'resource_major_problems':
                            remove_tag_data, sensor_suffix = False, 'fail'
                        else:
                            # tag_instance_mapping must be updated
                            remove_tag_data, sensor_suffix = self._try_evolve_ignoring_telematics(
                                tag_data, remove_tag_data, sensor_suffix
                            )
                    else:
                        remove_tag_data, sensor_suffix = False, 'fail'

                        if exc.error_code >= 500:
                            LOGGER.info('fatal error, pausing service, tag data: {}'.format(json.dumps(tag_data)))
                            remove_tag_data, sensor_suffix = False, 'fatal'
                except Exception as exc:
                    LOGGER.error('unknown error evolving scheduled tag {}'.format(tag_id))
                    remove_tag_data, sensor_suffix = False, 'fail'
            else:
                LOGGER.info('tag to evolve {} is absent'.format(tag_id))
                sensor_suffix = 'absent'

            if remove_tag_data:
                self._sync_helper.remove_extra_data('tags_to_evolve', tag_data)

            counters['scheduled_evolve.{}'.format(sensor_suffix)] += 1

            if sensor_suffix == 'fatal':
                pause_interval_seconds = self._sync_helper.get_extra_data('pause_interval_seconds', 5 * 60)
                self._sync_helper.pause(interval=datetime.timedelta(seconds=pause_interval_seconds))
                break

        self.solomon_helper.increment_counters(**counters)

    def _try_evolve_ignoring_telematics(self, tag_data, remove_tag_data, sensor_suffix):
        tag_id = tag_data['tag_id']

        tag_instance_mapping = (
            CarTag.objects.filter(tag_id=tag_id)
            .values('tag_id', 'tag', 'object_id', 'performer')
            .first()
        )

        if tag_instance_mapping is not None:
            car_id = tag_instance_mapping['object_id']

            try:
                with self._ignore_telematics(car_id, tag_id):
                    self.manager.evolve_tag(tag_data, tag_instance_mapping)

            except EvacuationException as exc:
                LOGGER.info('fatal error, pausing service, tag data: {}'.format(json.dumps(tag_data)))
                remove_tag_data, sensor_suffix = False, 'fatal'
        else:
            LOGGER.info('tag to evolve {} is absent'.format(tag_id))
            sensor_suffix = 'absent'

        return remove_tag_data, sensor_suffix

    @contextlib.contextmanager
    def _ignore_telematics(self, car_id, tag_id):
        service_comment = 'derived from tag {} to drop it after evolution'.format(tag_id)

        LOGGER.info('adding a service tag derived from {} for car {} to ignore telematics'.format(tag_id, car_id))

        self.manager.add_tag(
            car_id,
            self.TELEMATIC_PROBLEMS_TAG_INFO.name,
            self.TELEMATIC_PROBLEMS_TAG_INFO.priority,
            service_comment,
            check_exist=False
        )

        try:
            yield
        except EvacuationException as exc:
            LOGGER.error('error evolving scheduled tag {} ignoring telematics: {}'.format(tag_id, json.dumps(exc.data)))
            raise
        finally:
            self.manager.remove_tag(
                car_id,
                self.TELEMATIC_PROBLEMS_TAG_INFO.name,
                tag_filter=(lambda t: t.get('comment', '') == service_comment)
            )

            LOGGER.info('service tag for car {} to ignore telematics removed successfully'.format(car_id))
