import difflib
import logging
import re


LOGGER = logging.getLogger(__name__)


class TemplateMatcher(object):
    group_re = re.compile('{(\w+)\[(\w+)\]}')
    group_replacement = '{[]}'

    def __init__(self, template):
        self._sequence_matcher = difflib.SequenceMatcher(autojunk=False)

        self._groups = []
        self._template_lines = self.init_template(template)

        self._matches = None
        self._matched_assigned = None

    @property
    def matches(self):
        if len(self._groups) != self._matched_assigned:
            raise Exception('not all groups are matched')
        return self._matches

    @property
    def partial_matches(self):
        return self._matches

    def init_template(self, template):
        template = self.group_re.sub(self._perform_replacement, template)
        return template.splitlines()

    def _perform_replacement(self, match):
        self._groups.append(match.groups())
        return self.group_replacement

    def process_text(self, text, allow_partial=False):
        self._clear_matches()

        text_lines = text.splitlines()

        idx = None
        is_partial = False

        template_iter = (line for line in self._template_lines if line.strip())
        text_iter = (line for line in text_lines if line.strip())

        try:
            for idx, template_line, text_line in zip(range(len(self._template_lines)), template_iter, text_iter):
                self._sequence_matcher.set_seqs(template_line, text_line)
                for type_, _, _, right_start, right_end in self._sequence_matcher.get_opcodes():
                    if type_ != 'equal':
                        self._assign_next_match(text_line[right_start:right_end])
        except Exception:
            LOGGER.error('error matching text on line {}'.format(idx))
            if allow_partial:
                is_partial = True
            else:
                raise

        matches = self.matches if not allow_partial else self.partial_matches

        return matches, is_partial

    def _clear_matches(self):
        self._matches = {}
        self._matched_assigned = 0

    def _assign_next_match(self, substitution):
        group_names = self._groups[self._matched_assigned]

        root = self._matches
        for group_name in group_names[:-1]:
            root.setdefault(group_name, {})
            root = root[group_name]

        root[group_names[-1]] = substitution
        self._matched_assigned += 1
