import logging

from django.utils import timezone

import cars.settings
from cars.core.passport import AccountData  # for type hint
from cars.core.passport import PassportClient
from cars.users.core import UserProfileUpdater


LOGGER = logging.getLogger(__name__)


class IntroScreensUpdater:

    not_set = object()

    def __init__(self, user, passport_client):
        self._user = user
        self._user_profile_updater = UserProfileUpdater(self._user)
        self._passport_client = passport_client

        self._blackbox_info = None

    @classmethod
    def from_settings(cls, user):
        # use default blackbox and no TVM ticket (used on api)
        passport_client = PassportClient.from_settings(tvm_section=None)
        return cls(user=user, passport_client=passport_client)

    @classmethod
    def from_settings_admin_api(cls, user):
        # use external blackbox and corresponding tvm ticket (used on admin api)
        passport_client = PassportClient.from_settings_admin_api()
        return cls(user=user, passport_client=passport_client)

    def get_blackbox_info(self):
        """:rtype: AccountData"""
        if self._blackbox_info is None or self._blackbox_info.is_erroneous:
            self._blackbox_info = self._passport_client.get_account_public_info(self._user.uid)
        return self._blackbox_info

    def do_account_exists(self, default=True):
        blackbox_info = self.get_blackbox_info()
        exists = blackbox_info.get_property_safe('exists', default=default)
        return exists

    def maybe_update_username(self, *, actual_username=not_set):
        # try to update username to a display name from blackbox
        current_username = self._user.username

        if actual_username is self.not_set:
            blackbox_info = self.get_blackbox_info()
            # yandex account username (incl. generated for social ones)
            actual_username = blackbox_info.login

        if actual_username and actual_username != current_username:
            # replace generated or blank user name to current one
            LOGGER.info('updating username for user {} according to blackbox'.format(self._user.id))
            self._user_profile_updater.update_username(actual_username)

    def maybe_update_user_plus_status(self, *, actual_plus_status=not_set):
        current_plus_status = self._user.get_plus_status()

        if actual_plus_status is self.not_set:
            blackbox_info = self.get_blackbox_info()
            actual_plus_status = blackbox_info.get_property_safe('plus_status', default=None)

        if actual_plus_status is not None:
            if actual_plus_status != current_plus_status:
                LOGGER.info(
                    'plus status toggled for user {} according to blackbox, actual is {}'
                    .format(self._user.id, actual_plus_status)
                )

                self._user_profile_updater.update_plus_status(actual_plus_status)
        else:
            LOGGER.error('plus_status is None')

    def maybe_update_porsche_status(self):
        pass
