import datetime
import time

from django.utils import timezone

import cars.settings
from cars.eka.models import FuelCardActivation


class RefuelTimeChecker:

    UPDATE_THRESHOLD = 180.0

    def __init__(self, refuel_time_threshold, consider_fueled_threshold):
        self._refuel_time_threshold = refuel_time_threshold
        self._consider_fueled_theshold = consider_fueled_threshold
        self._recently_fueled = set()
        self._last_update = 0

    @classmethod
    def from_settings(cls):
        return cls(
            refuel_time_threshold=cars.settings.EKA['refuel_time_threshold'],
            consider_fueled_threshold=cars.settings.EKA['consider_fueled_threshold'],
        )

    def _update_cache(self):
        successful_refuelings = (
            FuelCardActivation.objects
            .filter(
                activated_at__gte=(
                    timezone.now() - datetime.timedelta(seconds=self._refuel_time_threshold)
                ),
                eventual_fuel_level__gte=self._consider_fueled_theshold,
            )
        )
        self._recently_fueled = set()
        for refueling in successful_refuelings:
            self._recently_fueled.add(str(refueling.car_id))
        self._last_update = time.time()

    def is_enough_time_passed_for_car(self, car):
        if time.time() - self._last_update > self.UPDATE_THRESHOLD:
            self._update_cache()
        return str(car.id) not in self._recently_fueled


REFUEL_TIME_CHECKER = RefuelTimeChecker.from_settings()
