import decimal
import logging

from django.utils import timezone
from rest_framework.serializers import (
    CharField, ModelSerializer, SerializerMethodField, UUIDField,
)

from cars.carsharing.core.segments_prototype import CarsharingSegmentsManagerPrototype
from cars.carsharing.models import CarModelSpecificationNew
from cars.carsharing.models.acceptance import CarsharingAcceptance
from cars.carsharing.models.car import Car
from cars.carsharing.models.car_location import CarLocation
from cars.carsharing.models.car_model import CarModel
from cars.carsharing.models.car_model_specification import CarModelSpecification
from cars.carsharing.models.car_telematics_state import CarTelematicsState
from cars.carsharing.models.parking import CarsharingParking
from cars.carsharing.models.reservation import CarsharingReservation
from cars.carsharing.models.reservation_paid import CarsharingReservationPaid
from cars.carsharing.models.ride import CarsharingRide
from cars.django.serializers import BaseSerializer, TimestampField


LOGGER = logging.getLogger(__name__)


class CarLocationSerializer(ModelSerializer):

    class Meta:
        model = CarLocation
        fields = [
            'course',
            'lat',
            'lon',
        ]


class CarSpecificationSerializer(ModelSerializer):

    class Meta:
        model = CarModelSpecification
        fields = [
            'name',
            'value',
        ]


class CarSpecificationNewSerializer(ModelSerializer):

    class Meta:
        model = CarModelSpecificationNew
        fields = [
            'name',
            'value',
        ]


class CarModelSerializer(ModelSerializer):

    specifications = SerializerMethodField()

    class Meta:
        model = CarModel
        fields = [
            'code',
            'name',
            'video_url',
            'image_large_url',
            'image_small_url',
            'image_map_url',  # Deprecate at some point.
            'image_map_url_2x',
            'image_map_url_3x',
            'image_pin_url_2x',
            'image_pin_url_3x',
            'fuel_type',
            'fuel_cap_side',
            'specifications',
        ]

    def get_specifications(self, obj):
        qs = obj.specifications_new.order_by('position')
        return CarSpecificationNewSerializer(qs, many=True).data


class CarTelematicsStateSerializer(ModelSerializer):

    fuel_endurance = SerializerMethodField()

    class Meta:
        model = CarTelematicsState
        fields = [
            'dipped_beam_on',
            'engine_temperature',
            'front_left_door_open',
            'front_right_door_open',
            'rear_left_door_open',
            'rear_right_door_open',
            'hood_open',
            'trunk_open',
            'fuel_level',
            'fuel_endurance',
            'hand_brake_on',
            'ignition_on',
            'engine_on',
            'mileage',
            'transmission_selector_park',
        ]

    def get_fuel_endurance(self, obj):  # pylint: disable=unused-argument
        return None


class SimpleCarSerializer(ModelSerializer):
    is_available = SerializerMethodField()
    cost_per_minute = SerializerMethodField()
    model = SerializerMethodField()
    location = CarLocationSerializer()
    telematics_state = CarTelematicsStateSerializer()
    props = SerializerMethodField()
    fuel_amount_required = SerializerMethodField()
    has_dvr = SerializerMethodField()

    class Meta:
        model = Car
        fields = [
            'id',
            'number',
            'is_available',
            'cost_per_minute',
            'location',
            'model',
            'telematics_state',
            'fuel_card_number',
            'props',
            'fuel_amount_required',
            'has_dvr',
        ]

    def get_has_dvr(self, obj):
        return obj.model.code in ('skoda_octavia', 'audi_a3', 'audi_q3')

    def get_fuel_amount_required(self, obj):
        if obj.get_telematics_state() is None:
            return None
        current_fuel_level = obj.get_telematics_state().fuel_level
        if current_fuel_level is None:
            return None

        fuel_tank_volume = obj.model.fuel_tank_volume
        if not fuel_tank_volume:
            fuel_tank_volume = 45.0

        required_amount = (100.0 - current_fuel_level) / 100.0 * fuel_tank_volume * 1.1

        return required_amount

    def get_is_available(self, obj):
        return obj.get_status() is Car.Status.AVAILABLE

    def get_props(self, obj):
        if 'props_per_car' not in self.context:
            return []

        try:
            props_per_car = self.context['props_per_car']
            if props_per_car is not None:
                return props_per_car[str(obj.id)]
        except Exception:
            LOGGER.exception(
                'car props serializer failed',
                extra={
                    'stack': True,
                },
            )

        return []

    def get_cost_per_minute(self, obj):  # pylint: disable=unused-argument
        ride_cost_per_minute = decimal.Decimal('7.99')
        parking_cost_per_minute = decimal.Decimal('2.99')
        free_parking = None
        try:
            tariffs_per_car = self.context['tariffs_per_car']
            if tariffs_per_car is not None:
                tariff = tariffs_per_car[obj]
                ride_cost_per_minute = float(tariff.ride_cost_per_minute)
                parking_cost_per_minute = float(tariff.parking_cost_per_minute)
                free_parking = tariff.free_parking
        except Exception:
            LOGGER.exception(
                'tariff serializer failed',
                extra={
                    'stack': True,
                },
            )

        cost_per_minute = {
            'carsharing_ride': ride_cost_per_minute,
            'carsharing_parking': parking_cost_per_minute,
            'carsharing_night_parking': {},
        }
        carsharing_night_parking = {}

        if parking_cost_per_minute == 0 and free_parking:
            free_parking_until = free_parking.end_date
            free_parking_next_cost = free_parking.next_tariff.parking_cost_per_minute
            cost_per_minute['carsharing_parking_free_until'] = free_parking_until.timestamp()
            cost_per_minute['carsharing_parking'] = float(free_parking_next_cost)

            # For design V2
            cost_per_minute['carsharing_night_parking_next_cost'] = float(free_parking_next_cost)
        else:
            cost_per_minute['carsharing_parking_free_until'] = None

            # For design V2
            cost_per_minute['carsharing_night_parking_next_cost'] = None

        # For design V2
        #
        # It is planned to be implemented at C++-backend. Anyway, if it becomes way too complicated, then
        # just move this part of logic to separate class of tariffs calculator

        segment_manager = CarsharingSegmentsManagerPrototype(obj)

        if segment_manager.has_night_waiting():
            is_plus_user = self.context.get('is_plus_user', False) or False
            timestamp = self.context.get('timestamp', 0)
            carsharing_night_parking['active'] = segment_manager.is_night_waiting_active(
                timestamp=timestamp,
                is_plus_user=is_plus_user,
            )

            # Probably remove segment_manager later, if no new requirements come
            carsharing_night_parking['cost'] = segment_manager.get_night_parking_cost()
            carsharing_night_parking['start'], carsharing_night_parking['end'] = (
                segment_manager.get_night_waiting_interval(timestamp, is_plus_user)
            )
        else:
            carsharing_night_parking = None

        cost_per_minute['carsharing_night_parking'] = carsharing_night_parking

        return cost_per_minute

    def get_model(self, obj):
        return {
            'code': obj.model_id,
        }


class CarForFineSerializer(ModelSerializer):
    model = SerializerMethodField()

    class Meta:
        model = Car
        fields = [
            'id',
            'number',
            'model',
        ]

    def get_model(self, obj):
        return {
            'code': obj.model_id,
        }


class CarListArgumentsSerializer(BaseSerializer):
    since = TimestampField(required=False)


class CarsharingAcceptanceOrderItemRequestSerializer(BaseSerializer):
    car_id = UUIDField()


class CarsharingAcceptanceOrderItemSerializer(ModelSerializer):
    car = SimpleCarSerializer()
    seconds_left = SerializerMethodField()

    class Meta:
        model = CarsharingAcceptance
        fields = [
            'car',
            'seconds_left',
        ]

    def get_seconds_left(self, obj):  # pylint: disable=unused-argument
        order_item = self.context['order_item']
        seconds_passed = (timezone.now() - order_item.started_at).total_seconds()
        seconds_left = max(5 * 60 - seconds_passed, 0)
        return seconds_left


class CarsharingParkingOrderItemRequestSerializer(BaseSerializer):
    car_id = UUIDField()


class CarsharingParkingOrderItemSerializer(ModelSerializer):

    car = SimpleCarSerializer()
    debt_termination = SerializerMethodField()

    class Meta:
        model = CarsharingParking
        fields = [
            'car',
            'debt_termination',
        ]

    def get_debt_termination(self, obj):
        seconds_left = self.get_debt_termination_seconds_left(obj)

        if seconds_left is None:
            debt_termination = None
        else:
            debt_termination = {
                'seconds_left': seconds_left,
                'debt_amount': float(self.context['debt_termination']['debt_amount']),
                'debt_threshold': float(self.context['debt_termination']['debt_threshold']),
            }

        return debt_termination

    def get_debt_termination_seconds_left(self, obj):
        if obj.debt_termination_max_duration_seconds is None:
            seconds_left = None
        else:
            order_item = self.context['order_item']
            seconds_passed = (timezone.now() - order_item.started_at).total_seconds()
            seconds_left = max(obj.debt_termination_max_duration_seconds - seconds_passed, 0)
        return seconds_left


class CarsharingReservationOrderItemRequestSerializer(BaseSerializer):
    car_id = UUIDField()


class CarsharingReservationOrderItemSerializer(ModelSerializer):
    car = SimpleCarSerializer()
    car_location = SerializerMethodField()
    seconds_left = SerializerMethodField()

    class Meta:
        model = CarsharingReservation
        fields = [
            'car',
            'car_location',
            'seconds_left',
        ]

    def get_car_location(self, obj):
        return {
            'course': obj.car_location_course,
            'lat': obj.car_location_lat,
            'lon': obj.car_location_lon,
        }

    def get_seconds_left(self, obj):
        order_item = self.context['order_item']
        seconds_passed = (timezone.now() - order_item.started_at).total_seconds()
        seconds_left = max(obj.max_duration_seconds - seconds_passed, 0)
        return seconds_left


class CarsharingReservationPaidOrderItemSerializer(ModelSerializer):
    car = SimpleCarSerializer()
    car_location = SerializerMethodField()

    class Meta:
        model = CarsharingReservationPaid
        fields = [
            'car',
            'car_location',
        ]

    def get_car_location(self, obj):
        return {
            'course': obj.car_location_course,
            'lat': obj.car_location_lat,
            'lon': obj.car_location_lon,
        }


class CarsharingRideOrderItemRequestSerializer(BaseSerializer):
    car_id = UUIDField()
    fix_id = CharField(required=False)


class CarsharingRideOrderItemSerializer(ModelSerializer):
    car = SimpleCarSerializer()
    can_finish = SerializerMethodField()

    class Meta:
        model = CarsharingRide
        fields = [
            'car',
            'mileage',
            'can_finish'
        ]

    def get_can_finish(self, obj):  # pylint: disable=unused-argument
        return True
