from rest_framework.serializers import FloatField, ModelSerializer, UUIDField, SerializerMethodField

from cars.django.serializers import BaseSerializer, TimestampField
from cars.fines.models import AutocodeFine, AutocodeFinePhoto

from .carsharing import CarForFineSerializer
from .order import SimpleOrderSerializer


class FinePhotoSerializer(ModelSerializer):

    class Meta:
        model = AutocodeFinePhoto
        fields = [
            'id',
            'url',
        ]


class FineSerializer(ModelSerializer):
    article_number_to_repr = {
        '12.09.2': 'Превышение скорости: немножко',
        '12.09.3': 'Превышение скорости: неплохо так',
        '12.09.4': 'Превышение скорости на 60—80 км/час',
        '12.9.6': 'Превышение скорости: неплохо так',
        '12.9.7': 'Превышение скорости на 60—80 км/час',
        '12.12.2': 'Выезд за стоп-линию',
        '12.12.3': 'Проезд на красный свет',
        '12.15.1': 'Проезд по обочине и пересечение автомобильной или пешей колонны',
        '12.15.4': 'Выезд на полосу встречного движения, либо на трамвайные пути встречного направления',
        '12.16.1': 'Не заметили знак или разметку (или всё сразу)',
        '12.16.2': 'Поворот налево или разворот в неположенном месте',
        '12.16.4': 'Остановка или брошенная машина в запрещенной знаком или разметкой зоне',
        '12.16.5': 'Не заметили знак или разметку, запрещающих остановку или стоянку машины',
        '12.17 ч.1.1': 'Играли в автобус или такси',
        '12.17 ч.1.2': 'Играли в автобус или такси',
        '12.19.2': 'Брошенная машина на парковке для инвалидов',
        '12.19 ч.5': 'Неправильная парковка',
        '12.19 ч.6': 'Остановка или брошенная машина посреди дороги или вторым рядом',
    }

    car = CarForFineSerializer()
    photos = FinePhotoSerializer(many=True, read_only=True)
    order = SerializerMethodField()

    article_koap_human_readable = SerializerMethodField()
    charge_passed_at = TimestampField()
    violation_time = TimestampField()

    sum_to_pay = FloatField()
    sum_to_pay_without_discount = FloatField()

    class Meta:
        model = AutocodeFine
        fields = [
            'id',
            'article_koap',
            'article_koap_human_readable',
            'charge_passed_at',
            'ruling_number',
            'ruling_date',
            'violation_time',
            'violation_place',
            'violation_latitude',
            'violation_longitude',
            'sum_to_pay_without_discount',
            'sum_to_pay',
            'car',
            'order',
            'photos',
        ]

    def get_article_koap_human_readable(self, fine):
        article_number = fine.article_koap.split(' - ')[0]
        return self.article_number_to_repr.get(article_number, fine.article_koap)

    def get_order(self, fine):
        if fine.session_id:
            return {
                'id': fine.session_id,
                'created_at': None,
                'completed_at': None,
            }

        return SimpleOrderSerializer(fine.order).data
