from rest_framework.fields import empty, UUIDField, CharField, BooleanField, ChoiceField
from rest_framework.exceptions import ValidationError
from rest_framework.serializers import as_serializer_error

from cars.django.serializers import BaseSerializer, TimestampField
from cars.core.util import datetime_helper

from cars.settings import INCIDENT

from cars.carsharing.models import Car
from cars.users.models import User


class CarInfoSerializer(BaseSerializer):
    id = UUIDField(required=True)
    add_tag = BooleanField(default=True)

    def run_validation(self, data=empty):
        try:
            value = super().run_validation(data)
            self._coerce_car(value)

        except ValidationError as original_exc:
            raise ValidationError(detail=as_serializer_error(original_exc))

        return value

    def _coerce_car(self, data):
        car_id = str(data['id'])
        car = Car.objects.filter(id=car_id).first()
        if car is None:
            raise ValidationError(detail='no car with id "{}" exists'.format(car_id))

        data['number'] = car.number
        data['vin'] = car.vin
        data['ctc'] = car.registration_id
        data['model'] = car.model.name
        data['link'] = '{}/#/cars/{}/info'.format(INCIDENT['host'], car_id)


class UserInfoSerializer(BaseSerializer):
    DEFAULT_PROPERTY_VALUE = '**Нет информации**'

    FIELDS_TO_ADD = ('first_name', 'email', 'phone')

    id = UUIDField(required=False, default=None, allow_null=True)

    def run_validation(self, data=empty):
        data = data if data is not empty else {}  # to set defaults

        try:
            value = super().run_validation(data)
            self._coerce_user(value)

        except ValidationError as original_exc:
            raise ValidationError(detail=as_serializer_error(original_exc))

        return value

    def _coerce_user(self, data):
        is_default = (data['id'] is None)

        if not is_default:
            user_id = str(data['id'])
            user = User.objects.filter(id=user_id).first()
            if user is None:
                raise ValidationError(detail='no user with id "{}" exists'.format(user_id))
        else:
            user_id = self.DEFAULT_PROPERTY_VALUE
            user = None

        for field in self.FIELDS_TO_ADD:
            value = getattr(user, field, None)
            data[field] = str(value) if value else self.DEFAULT_PROPERTY_VALUE

        data['id'] = user_id
        data['is_default'] = is_default

        if not is_default:
            data['print_name'] = user.get_full_name()
            data['link'] = '{}/#/clients/{}/info'.format(INCIDENT['host'], user_id)
        else:
            data['print_name'] = self.DEFAULT_PROPERTY_VALUE
            data['link'] = self.DEFAULT_PROPERTY_VALUE


class SessionInfoSerializer(BaseSerializer):
    DEFAULT_PROPERTY_VALUE = '**Нет информации**'

    id = CharField(required=False, default=None, allow_null=True)
    order_end_time = CharField(required=False, default=DEFAULT_PROPERTY_VALUE)

    def run_validation(self, data=empty):
        data = data if data is not empty else {}  # to set defaults

        try:
            value = super().run_validation(data)
            self._coerce_session(value)

        except ValidationError as original_exc:
            raise ValidationError(detail=as_serializer_error(original_exc))

        return value

    def _coerce_session(self, data):
        is_default = (data['id'] is None)

        data['is_default'] = is_default

        if not is_default:
            session_id = str(data['id'])
            data['id'] = session_id
            data['link'] = '{}/#/session/{}/'.format(INCIDENT['host'], session_id)
        else:
            data['id'] = self.DEFAULT_PROPERTY_VALUE
            data['link'] = self.DEFAULT_PROPERTY_VALUE


class EvacuationDetailsSerializer(BaseSerializer):
    violation_paragraph = CharField(default='', allow_blank=True)

    datetime = CharField(default='', allow_blank=True)
    timestamp = TimestampField(default=None, allow_null=True)
    address = CharField(default='', allow_blank=True)

    city = ChoiceField(choices=list(INCIDENT['evacuation']['ticket_queue']))

    inspection_department = CharField(default='', allow_blank=True)
    inspection_department_address = CharField(default='', allow_blank=True)
    inspection_department_phone = CharField(default='', allow_blank=True)
    inspection_parking_address = CharField(default='', allow_blank=True)

    def run_validation(self, data=empty):
        data = data if data is not empty else {}  # to set defaults

        try:
            value = super().run_validation(data)
            self._coerce_datetime(value)

        except ValidationError as original_exc:
            raise ValidationError(detail=as_serializer_error(original_exc))

        return value

    def _coerce_datetime(self, data):
        evacuation_date = data.get('timestamp')

        if evacuation_date is not None:
            data['datetime'] = datetime_helper.change_timezone(evacuation_date).strftime('%d.%m.%Y %H:%M:%S')


class EvacuationInfoSerializer(BaseSerializer):
    car = CarInfoSerializer()
    user = UserInfoSerializer()
    session = SessionInfoSerializer()
    evacuation = EvacuationDetailsSerializer()
