import time

from rest_framework.serializers import SerializerMethodField

import cars.settings
from cars.billing.models.bonus_account import BonusAccount
from cars.core.util import import_class
from cars.django.serializers import BaseSerializer
from cars.orders.core.preliminary_payments_manager import OrderPreliminaryPaymentsManager
from cars.orders.models.order import Order
from .order import OrderSerializer
from .user import UserSerializer


class IntroSerializer(BaseSerializer):
    user = UserSerializer()
    order = SerializerMethodField()
    preliminary_payments = SerializerMethodField()
    registration = SerializerMethodField()
    server_time = SerializerMethodField()
    user_setup = SerializerMethodField()

    def get_order(self, request):
        try:
            order = (
                Order.objects
                .with_related()
                .get(
                    user=request.user,
                    completed_at__isnull=True,
                )
            )
        except Order.DoesNotExist:
            order = None
        data = OrderSerializer(order).data if order else None
        return data

    def get_preliminary_payments(self, request):
        manager = OrderPreliminaryPaymentsManager.from_settings()
        amount = manager.get_default_amount(request.user)
        enabled = amount > 0
        data = {
            'enabled': enabled,
            'amount': float(amount) if enabled else None,
        }
        return data

    def get_registration(self, request):
        user = request.user

        bonus_account = user.get_bonus_account()
        if bonus_account is None:
            bonus_account = BonusAccount(user=user)

        return {
            'taxi_cashback': float(bonus_account.balance),
        }

    def get_server_time(self, request):  # pylint: disable=unused-argument
        return time.time()

    def get_user_setup(self, request):
        user_setup_class = import_class(cars.settings.DRIVE['intro']['user_setup_class'])
        user_setup = user_setup_class.from_request(request)
        data = {
            'email': {
                'suggested': user_setup.email_suggested,
                'verified': user_setup.email_verified,
            },
            'phone': {
                'suggested': user_setup.phone_suggested,
                'verified': user_setup.phone_verified,
            },
        }
        return data
