import copy
import logging
import time

from rest_framework.serializers import (
    CharField, ChoiceField, DictField, ListField, ModelSerializer,
    NullBooleanField, SerializerMethodField, ValidationError,
)

import cars.settings
from cars.calculator.core.client import CalculatorClient
from cars.carsharing.models import CarsharingAcceptance
from cars.django.serializers import BaseSerializer, TimestampField
from cars.drive.core.refuel_time_checker import REFUEL_TIME_CHECKER
from cars.eka.models import FuelCardActivation
from cars.orders.core.order_item_managers.carsharing_acceptance import CarsharingAcceptanceManager
from cars.orders.core.order_debt_manager import OrderDebtManager
from cars.orders.core.order_payment_processor import OrderPaymentProcessor
from cars.orders.core.push_client import PUSH_CLIENT
from cars.orders.models.order import Order
from cars.orders.models.order_item import OrderItem
from cars.orders.models.order_item_tariff import OrderItemTariff
from cars.orders.models.order_payment_method import OrderPaymentMethod
from .carsharing import (
    CarsharingAcceptanceOrderItemRequestSerializer,
    CarsharingAcceptanceOrderItemSerializer,
    CarsharingParkingOrderItemRequestSerializer,
    CarsharingParkingOrderItemSerializer,
    CarsharingReservationOrderItemRequestSerializer,
    CarsharingReservationOrderItemSerializer,
    CarsharingReservationPaidOrderItemSerializer,
    CarsharingRideOrderItemRequestSerializer,
    CarsharingRideOrderItemSerializer,
)


LOGGER = logging.getLogger(__name__)


class OrderItemActionParamsPhotoSerializer(BaseSerializer):
    id = CharField(required=True)
    content = CharField(required=True)
    mime_type = CharField(required=True)


class OrderItemActionParamsCarSerializer(BaseSerializer):
    car_condition = ChoiceField(
        choices=[x.value for x in CarsharingAcceptance.CarCondition],
        required=True
    )
    fuel_card_present = NullBooleanField(required=True)
    insurance_present = NullBooleanField(required=True)
    sts_present = NullBooleanField(required=True)
    comment = CharField(required=False)


class OrderItemRequestSerializer(BaseSerializer):
    type = ChoiceField(choices=[x.value for x in OrderItem.Type])
    params = None  # Specified in to_internal_value().

    def to_internal_value(self, data):
        try:
            type_ = OrderItem.Type(data['type'])
        except (KeyError, ValueError):
            raise ValidationError(
                detail={
                    'type': 'type.invalid',
                },
            )

        if type_ is OrderItem.Type.CARSHARING_ACCEPTANCE:
            params_serializer = CarsharingAcceptanceOrderItemRequestSerializer()
        elif type_ is OrderItem.Type.CARSHARING_PARKING:
            params_serializer = CarsharingParkingOrderItemRequestSerializer()
        elif type_ is OrderItem.Type.CARSHARING_RESERVATION:
            params_serializer = CarsharingReservationOrderItemRequestSerializer()
        elif type_ is OrderItem.Type.CARSHARING_RIDE:
            params_serializer = CarsharingRideOrderItemRequestSerializer()
        else:
            raise RuntimeError('unreachable: {}'.format(type_))

        self.fields['params'] = params_serializer

        return super().to_internal_value(data)


class OrderPaymentMethodSerializer(BaseSerializer):

    type = ChoiceField(choices=[x.value for x in OrderPaymentMethod.Type])
    params = DictField()

    def to_representation(self, obj):
        data = super().to_representation(obj)

        type_ = obj.get_type()
        if type_ is OrderPaymentMethod.Type.CARD:
            params = {
                'paymethod_id': obj.card_paymethod_id,
            }
        else:
            LOGGER.warning('unreachable: %s', type_)
            params = {}
        data['params'] = params

    def to_internal_value(self, data):
        value = super().to_internal_value(data)
        transformed_value = {
            'type': value['type'],
            'card_paymethod_id': value['params'].get('paymethod_id'),
        }
        return transformed_value


class OrderRequestSerializer(BaseSerializer):
    payment_method = OrderPaymentMethodSerializer(required=False)
    items = ListField(child=OrderItemRequestSerializer())


class OrderItemDetailsArgumentsSerializer(BaseSerializer):
    action = CharField()

    def to_internal_value(self, data):
        try:
            action_ = data['action']
        except KeyError:
            raise ValidationError(
                detail={
                    'type': 'action.missing_field',
                },
            )

        params_serializer = None

        # process only order items with non-empty params
        if action_ == CarsharingAcceptanceManager.ActionTypes.REPORT_CONDITION.value:
            params_serializer = OrderItemActionParamsCarSerializer()
        elif action_ == CarsharingAcceptanceManager.ActionTypes.REPORT_PHOTO.value:
            params_serializer = OrderItemActionParamsPhotoSerializer()

        if params_serializer is not None:
            self.fields['params'] = params_serializer

        return super().to_internal_value(data)


class OrderItemSerializer(ModelSerializer):

    order_debt_manager = OrderDebtManager.from_settings(push_client=PUSH_CLIENT)
    order_payment_processor = OrderPaymentProcessor.from_settings()

    started_at = TimestampField()
    finished_at = TimestampField()
    cost = SerializerMethodField()
    params = SerializerMethodField()

    class Meta:
        model = OrderItem
        fields = [
            'id',
            'type',
            'started_at',
            'finished_at',
            'cost',
            'params',
        ]

    def get_cost(self, obj):
        summary = self.order_payment_processor.get_order_item_payment_summary(obj)
        return {
            'value': float(summary.cost),
            'bonus_value': float(summary.bonus_amount),
            'card_value': float(summary.card_amount),
            'plus': self.get_plus(obj),
        }

    def get_plus(self, obj):
        has_plus_discount = obj.order.has_plus_discount or False
        summary = self.order_payment_processor.get_order_item_payment_summary(obj)
        if has_plus_discount:
            discount = cars.settings.CARSHARING['plus']['discount_multiplier']
            discounted_value = round(float(summary.cost) / float(discount) - float(summary.cost), 2)
            return {
                'is_plus_discounted': True,
                'discounted_value': discounted_value,
                'full_price': float(summary.cost) + discounted_value,
                'discount': int((1.0 - float(discount)) * 100 + 0.1),  # +0.1 to avoid int(0.58 * 100) effect
            }
        else:
            return {
                'is_plus_discounted': False,
                'discounted_value': 0.0,
                'full_price': float(summary.cost),
                'discount': 0,
            }

    def get_params(self, obj):
        context = copy.deepcopy(self.context)
        context['order_item'] = obj

        obj_type = obj.get_type()
        if obj_type is OrderItem.Type.CARSHARING_ACCEPTANCE:
            serializer_class = CarsharingAcceptanceOrderItemSerializer
            impl = obj.carsharing_acceptance
        elif obj_type is OrderItem.Type.CARSHARING_PARKING:
            serializer_class = CarsharingParkingOrderItemSerializer
            impl = obj.carsharing_parking
            context['debt_termination'] = {
                'debt_amount': self.order_debt_manager.get_debt(obj.order.user),
                'debt_threshold': self.order_debt_manager.debt_order_termination_threshold,
            }
        elif obj_type is OrderItem.Type.CARSHARING_RESERVATION:
            serializer_class = CarsharingReservationOrderItemSerializer
            impl = obj.carsharing_reservation
        elif obj_type is OrderItem.Type.CARSHARING_RESERVATION_PAID:
            serializer_class = CarsharingReservationPaidOrderItemSerializer
            impl = obj.carsharing_reservation_paid
        elif obj_type is OrderItem.Type.CARSHARING_RIDE:
            serializer_class = CarsharingRideOrderItemSerializer
            impl = obj.carsharing_ride
        else:
            raise RuntimeError('unreachable: {}'.format(obj_type))

        serializer = serializer_class(impl, context=context)

        return serializer.data


class OrderSerializer(ModelSerializer):

    order_payment_processor = OrderPaymentProcessor.from_settings()

    created_at = TimestampField()
    completed_at = TimestampField()
    items = SerializerMethodField()
    cost = SerializerMethodField()

    can_fuel = SerializerMethodField()
    fuel_card_active = SerializerMethodField()

    server_time = SerializerMethodField()

    class Meta:
        model = Order
        fields = [
            'id',
            'created_at',
            'completed_at',
            'items',
            'cost',
            'can_fuel',
            'fuel_card_active',
            'server_time',
        ]

    def get_can_fuel(self, obj):
        """
        Boolean value meaning whether a user can fuel the car using fuel card or not.
        """
        acceptance = obj.items.filter(type=OrderItem.Type.CARSHARING_ACCEPTANCE.value).first()
        if acceptance is None:
            return False

        car = acceptance.get_impl().car
        fuel_card_present_in_model = car.fuel_card_number is not None
        car_insufficiently_fueled = (
            car.telematics_state.fuel_level is not None and
            car.telematics_state.fuel_level <= cars.settings.EKA['refuel_threshold']
        )

        car_model_ok_for_refuel = car.model.manufacturer != 'Porsche'
        enough_refuel_time_passed = REFUEL_TIME_CHECKER.is_enough_time_passed_for_car(car)

        return (
            fuel_card_present_in_model and
            car_insufficiently_fueled and
            car_model_ok_for_refuel and
            enough_refuel_time_passed
        )

    def get_fuel_card_active(self, obj):
        activation = (
            FuelCardActivation.objects.filter(
                order=obj,
                blocked_at__isnull=True,
            )
            .first()
        )
        return activation is not None

    def get_cost(self, obj):
        summary = self.order_payment_processor.get_order_payment_summary(obj)
        return {
            'value': float(summary.cost),
            'bonus_value': float(summary.bonus_amount),
            'card_value': float(summary.card_amount),
            'plus': self.get_plus(obj),
        }

    def get_plus(self, obj):
        has_plus_discount = obj.has_plus_discount or False
        summary = self.order_payment_processor.get_order_payment_summary(obj)
        if has_plus_discount:
            discount = cars.settings.CARSHARING['plus']['discount_multiplier']
            discounted_value = round(float(summary.cost) / float(discount) - float(summary.cost), 2)
            return {
                'is_plus_discounted': True,
                'discounted_value': discounted_value,
                'full_price': float(summary.cost) + discounted_value,
                'discount': int((1.0 - float(discount)) * 100 + 0.1),  # +0.1 to avoid int(0.58 * 100) effect
            }
        else:
            return {
                'is_plus_discounted': False,
                'discounted_value': 0.0,
                'full_price': float(summary.cost),
                'discount': 0,
            }

    def get_items(self, obj):
        ride_cost_per_minute = 0
        parking_cost_per_minute = 0

        for item in reversed(obj.get_sorted_items()):
            if item.finished_at is not None:
                continue

            if item.tariff is None or item.tariff.get_type() is not OrderItemTariff.Type.PER_MINUTE:
                continue

            cost_per_minute = item.tariff.per_minute_params.cost_per_minute

            if item.get_type() is OrderItem.Type.CARSHARING_RIDE:
                ride_cost_per_minute = cost_per_minute
            else:
                parking_cost_per_minute = cost_per_minute

        car = obj.get_sorted_items()[0].get_impl().car
        context = {
            'tariffs_per_car': {
                car: CalculatorClient.CarsharingBaseTariff(
                    start_time=None,
                    ride_cost_per_minute=ride_cost_per_minute,
                    parking_cost_per_minute=parking_cost_per_minute,
                    free_parking=None,
                )
            },
        }

        return [
            OrderItemSerializer(item, context=context).data
            for item in obj.get_sorted_items()
        ]

    def get_server_time(self, obj):  # pylint: disable=unused-argument
        return time.time()


class SimpleOrderSerializer(ModelSerializer):

    created_at = TimestampField()
    completed_at = TimestampField()

    class Meta:
        model = Order
        fields = [
            'id',
            'created_at',
            'completed_at',
        ]
