from rest_framework.serializers import ModelSerializer, SerializerMethodField

from cars.billing.models.bonus_account import BonusAccount
from cars.callcenter.models.call_priority import CallPriorityUser
from cars.orders.core.order_debt_manager import OrderDebtManager
from cars.users.models.user import User
from cars.users.models.user_intro_screens import UserIntroScreens
from ..core.push_client import PUSH_CLIENT
from .bonus_account import BonusAccountSerializer
from .user_credit_card import UserCreditCardSerializer


class UserIntroScreensSerializer(ModelSerializer):

    plus = SerializerMethodField()
    porsche = SerializerMethodField()

    class Meta:
        model = UserIntroScreens
        fields = [
            'plus',
            'porsche'
        ]

    def get_plus(self, obj):
        is_qualified_for = obj.plus_screen_qualified_for_at is not None
        is_shown = obj.plus_screen_viewed_at is not None
        return is_qualified_for and (not is_shown)

    def get_porsche(self, obj):
        is_qualified_for = obj.porsche_screen_qualified_for_at is not None
        is_shown = obj.porsche_screen_viewed_at is not None
        return is_qualified_for and (not is_shown)


class UserSerializer(ModelSerializer):

    bonus_account = SerializerMethodField()
    credit_card = UserCreditCardSerializer()
    call_priority = SerializerMethodField()
    debt = SerializerMethodField()
    plus = SerializerMethodField()
    introscreens = SerializerMethodField()

    class Meta:
        model = User
        fields = [
            'id',
            'uid',
            'username',
            'email',
            'phone',
            'status',
            'first_name',
            'last_name',
            'patronymic_name',
            'bonus_account',
            'call_priority',
            'credit_card',
            'debt',
            'plus',
            'introscreens',
        ]

    def get_call_priority(self, obj):
        return bool(CallPriorityUser.objects.filter(user_id=obj.id).first())

    def get_bonus_account(self, obj):
        bonus_account = obj.get_bonus_account()
        if bonus_account is None:
            bonus_account = BonusAccount(user=obj)
        data = BonusAccountSerializer(bonus_account).data
        return data

    def get_debt(self, obj):
        order_debt_manager = OrderDebtManager.from_settings(push_client=PUSH_CLIENT)
        value = order_debt_manager.get_debt(user=obj)
        return {
            'value': float(value),
        }

    def get_plus(self, obj):
        return {
            'is_activated': obj.get_plus_status(),
            'is_viewed': obj.is_plus_screen_viewed or False,
        }

    def get_introscreens(self, obj):
        introscreens = obj.introscreens
        if introscreens is None:
            introscreens = UserIntroScreens()
        return UserIntroScreensSerializer(introscreens).data
