import copy
import time
import uuid

from cars.registration.core.registration_manager import RegistrationManager
from django.utils import timezone
from rest_framework.exceptions import NotFound, PermissionDenied
from rest_framework.mixins import ListModelMixin
from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST, HTTP_403_FORBIDDEN

import cars.settings
from cars.carsharing.core.parking_area_checker import ParkingAreaChecker
from cars.carsharing.core.props_collector import CarPropsCollector
from cars.carsharing.core.segments_prototype import CarsharingSegmentsManagerPrototype
from cars.carsharing.core.telematics_proxy import TelematicsApiResponse
from cars.carsharing.models.car import Car
from cars.carsharing.models.car_model import CarModel
from cars.core.util import import_class
from cars.users.models import User
from ..core.car_commander import CarCommander
from ..serializers.carsharing import (
    CarListArgumentsSerializer, CarModelSerializer, SimpleCarSerializer,
)
from .base import DriveAPIView


class CarListView(DriveAPIView):

    arguments_serializer_class = CarListArgumentsSerializer

    calculator = import_class(cars.settings.CALCULATOR['client']['class']).from_settings()
    car_props_collector = CarPropsCollector()

    def do_get(self, request):
        user_status = request.user.get_status()
        is_not_registered = user_status in RegistrationManager.USER_REGISTRATION_STATUSES
        if is_not_registered:
            return Response(status=HTTP_403_FORBIDDEN)

        if request.user.status not in (User.Status.ACTIVE.value, User.Status.DEBT.value):
            return Response(
                {
                    'cars': [],
                    'server_time': time.time(),
                }
            )

        carsharing_settings = cars.settings.DRIVE['carsharing']

        server_time = time.time()

        since = self.request.arguments.get('since')

        cars_qs = (
            Car.objects
            .using(cars.settings.DB_RO_ID)
            .select_related('location', 'model', 'telematics_state')
        )
        if not cars.settings.IS_TESTS:
            cars_qs = (
                cars_qs.filter(
                    model__code__in=[
                        'renault_kaptur',
                        'kia_rio',
                        'kia_rio_xline',
                        'skoda_octavia',
                    ]
                )
            )

        location_updated_at_max_delta = (
            carsharing_settings['car_list_location_updated_at_max_delta']
        )
        if location_updated_at_max_delta:
            min_location_updated_at = timezone.now() - location_updated_at_max_delta
            cars_qs = cars_qs.filter(location__updated_at__gte=min_location_updated_at)

        if since:
            threshold = carsharing_settings['car_list_since_threshold']
            if timezone.now() - since > threshold:
                return Response(status=HTTP_400_BAD_REQUEST)
            cars_qs = cars_qs.filter(updated_at__gte=since)
        else:
            cars_qs = cars_qs.filter(status=Car.Status.AVAILABLE.value)

        context = super().get_serializer_context()
        context['tariffs_per_car'] = self.calculator.get_carsharing_base_tariffs(
            user=request.user,
            cars=cars_qs,
        )
        context['props_per_car'] = self.car_props_collector.get_props(cars_qs)
        context['is_plus_user'] = request.user.get_plus_status()
        context['timestamp'] = int(server_time)

        cars_list = SimpleCarSerializer(cars_qs, many=True, context=context).data
        cars_list_filtered = CarsharingSegmentsManagerPrototype.filter_cars_list(
            cars_list,
            request.user
        )

        data = {
            'cars': cars_list_filtered,
            'server_time': server_time,
        }

        return Response(data)


class BaseCarCommandView(DriveAPIView):

    _commander = None

    @classmethod
    def _get_commander(cls):
        if cls._commander is None:
            cls._commander = CarCommander.from_settings()
        return cls._commander

    def do_post(self, request, car_id):
        try:
            car_id = uuid.UUID(car_id)
        except ValueError:
            return Response(status=HTTP_400_BAD_REQUEST)

        try:
            car = Car.objects.get(id=car_id)
        except Car.DoesNotExist:
            raise NotFound

        commander = self._get_commander()

        error = None
        try:
            self._do(
                user=request.user,
                car=car,
                commander=commander,
            )
        except TelematicsApiResponse.Error as e:
            assert e.code is not None
            error = 'car.{}'.format(e.code)
        except commander.NotAuthorized:
            raise PermissionDenied
        except commander.NotFound:
            raise NotFound

        context = {
            'tariffs_per_car': None,
        }
        data = SimpleCarSerializer(  # Place car data in document root for compatibility.
            car,
            context=context,
        ).data
        data['car'] = copy.deepcopy(data)
        if error:
            data['status'] = 'errors'
            data['errors'] = [error]
            return Response(
                data=data,
            )
        else:
            data['status'] = 'success'

        return Response(data)

    def _do(self, user, car, commander):
        raise NotImplementedError


class BlinkCarCommandView(BaseCarCommandView):
    def _do(self, user, car, commander):
        commander.blink(user=user, car=car)


class WarmupCommandView(BaseCarCommandView):
    def _do(self, user, car, commander):
        commander.warmup(user=user, car=car)


class CarModelListView(ListModelMixin, DriveAPIView):

    serializer_class = CarModelSerializer

    def get_queryset(self):
        return CarModel.objects.all().prefetch_related('specifications_new')

    def do_get(self, request, *args, **kwargs):
        if request.user.status not in (User.Status.ACTIVE.value, User.Status.DEBT.value):
            return Response([])

        return self.list(request, *args, **kwargs)


class ParkingAreaView(DriveAPIView):

    _parking_area_checker = ParkingAreaChecker.from_settings()
    POI_JSON = [
        # airports
        {
            'area_index': 13,
            'name': 'SVO',
            'type': 'airport',
            'tooltip': {
                'title': 'Аэропорт «Шереметьево»',
                'message': 'Парковка напротив терминала D, второй этаж',
                'url': 'https://yandex.ru/support/drive/airports/svo.html',
            },
            'center': [
                55.961089530817176,
                37.40640987244209
            ]
        },
        {
            'area_index': 30,
            'name': 'VKO',
            'type': 'airport',
            'tooltip': {
                'title': 'Аэропорт «Внуково»',
                'message': 'Парковки напротив терминалов D и A, четвертый этаж',
                'url': 'https://yandex.ru/support/drive/airports/vko.html'
            },
            'center': [
                55.605019311304304,
                37.29045748975283
            ]
        },
        # other
        {
            'area_index': 1,
            'name': 'Долгопрудный',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.93932321483856,
                37.51112178789601
            ]
        },
        {
            'area_index': 2,
            'name': 'Красногорск',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.82646335880991,
                37.31570058777952
            ]
        },
        {
            'area_index': 3,
            'name': 'Мытищи',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.91148144839785,
                37.72897490544872
            ]
        },
        {
            'area_index': 4,
            'name': 'Ново-Переделкино',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.64765806695256,
                37.37814751241665
            ]
        },
        {
            'area_index': 5,
            'name': 'Бизнес-парк К2',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.59162390673066,
                37.474609774282996
            ]
        },
        {
            'area_index': 6,
            'name': 'Северный',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.92753989669045,
                37.54278020916888
            ]
        },
        {
            'area_index': 7,
            'name': 'Саларьево',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.622587209619155,
                37.424920897369674
            ]
        },
        {
            'area_index': 8,
            'name': 'Коммунарка',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.57135900603167,
                37.4790908276817
            ]
        },
        {
            'area_index': 9,
            'name': 'Южное Бутово',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.542375707532855,
                37.53531573730921
            ]
        },
        {
            'area_index': 10,
            'name': 'Северное Бутово',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.56733384511904,
                37.57603025963406
            ]
        },
        {
            'area_index': 12,
            'name': 'Котельники',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.66448722271069,
                37.861773419571605
            ]
        },
        {
            'area_index': 14,
            'name': 'Яндекс.Драйв',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.651592856988586,
                37.440585385699684
            ]
        },
        {
            'area_index': 17,
            'name': 'Мега Теплый Стан',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.60416542968995,
                37.49278402396295
            ]
        },
        {
            'area_index': 18,
            'name': 'Major City',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.79009365066467,
                37.253866196924925
            ]
        },
        {
            'area_index': 21,
            'name': 'ТРЦ Vegas',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.586592848252955,
                37.721960902390265
            ]
        },
        {
            'area_index': 22,
            'name': 'Outlet Village Белая Дача',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.660353650872786,
                37.88105647777749
            ]
        },
        {
            'area_index': 24,
            'name': 'Люберцы',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.68409333393663,
                37.86927064355715
            ]
        },
        {
            'area_index': 25,
            'name': 'Новокосино',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.742744027594206,
                37.86265034731225
            ]
        },
        {
            'area_index': 27,
            'name': 'Мега Белая дача',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.653873548049006,
                37.84407594215698
            ]
        },
        {
            'area_index': 32,
            'name': 'Поселок Внуково',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.61032212297964,
                37.296098918098416
            ]
        },
        {
            'area_index': 33,
            'name': 'Московский',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.59792750254694,
                37.35904087819301
            ]
        },
        {
            'area_index': 34,
            'name': 'Путилково',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.87454759357025,
                37.39481328705024
            ]
        },
        {
            'area_index': 36,
            'name': 'Химки',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.89616077034004,
                37.407171421943026
            ]
        },
        {
            'area_index': 37,
            'name': 'Митино',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.84698744115025,
                37.36213151473121
            ]
        },
        {
            'area_index': 38,
            'name': 'Микрогород в лесу',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.87040519593015,
                37.326688780753834
            ]
        },
        {
            'area_index': 39,
            'name': 'Крокус Экспо',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.82314809171373,
                37.38825929367636
            ]
        },
        {
            'area_index': 40,
            'name': 'Павшинская пойма',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.81443934183835,
                37.35455519676182
            ]
        },
        {
            'area_index': 41,
            'name': 'Левобережный',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.894808995054284,
                37.4787045369537
            ]
        },
        {
            'area_index': 42,
            'name': 'Минск DEV-office',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                53.890797459132976,
                27.525955636554663
            ]
        },
        {
            'area_index': 44,
            'name': 'Некрасовка',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.699780193895414,
                37.908313080984
            ]
        },
        {
            'area_index': 46,
            'name': 'Реутов',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.759448799153546,
                37.85618793484267
            ]
        },
        {
            'area_index': 48,
            'name': 'ЖК Митино Парк',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.83521819079043,
                37.35283577550763
            ]
        },
        {
            'area_index': 49,
            'name': 'Королёв',
            'type': 'ordinary',
            'tooltip': None,
            'center': [
                55.91726072201628,
                37.857021103468036
            ]
        },
    ]

    def do_get(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        data = {
            'parking_area': self._parking_area_checker.get_parking_area_geojson(),
            'poi': self.POI_JSON,
        }
        return Response(data)
