import logging
import time

from rest_framework.exceptions import NotFound
from rest_framework.mixins import ListModelMixin, RetrieveModelMixin
from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST

import cars.settings
from cars.django.pagination import Pagination
from cars.django.util import ReqAnsLogger
from cars.orders.core.order_item_managers.base import BaseOrderItemManager, OrderItemActionContext
from cars.orders.core.order_manager import OrderManager
from cars.orders.core.order_request import (
    OrderItemRequest, OrderItemRequestContext, OrderPaymentMethodRequest, OrderRequest,
)
from cars.orders.models.order import Order
from cars.orders.models.order_item import OrderItem
from .base import DriveAPIView
from ..core.push_client import PUSH_CLIENT
from ..serializers.order import (
    OrderItemDetailsArgumentsSerializer,
    OrderItemRequestSerializer,
    OrderItemSerializer,
    OrderRequestSerializer,
    OrderSerializer,
)


LOGGER = logging.getLogger(__name__)


class OrderDetailsView(RetrieveModelMixin, DriveAPIView):

    lookup_url_kwarg = 'order_id'
    serializer_class = OrderSerializer

    def get_queryset(self):
        return Order.objects.filter(user=self.request.user)

    def get_object(self):
        order = super().get_object()
        order.prefetch_related_objects()
        return order

    def do_get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, **kwargs)


class OrderListView(ListModelMixin, DriveAPIView):

    pagination_class = Pagination
    serializer_class = OrderSerializer

    arguments_serializer_class = OrderRequestSerializer

    order_manager = OrderManager.from_settings(push_client=PUSH_CLIENT)

    def do_get(self, *args, **kwargs):
        return self.list(*args, **kwargs)

    def do_post(self, request):
        # Temporary workaround for Android client not sending /session requests.
        context = OrderItemRequestContext(
            oauth_token=request.oauth_token,
        )

        handled_error_types = (
            OrderItemRequest.Error,
            OrderManager.Error,
            OrderPaymentMethodRequest.Error,
            OrderRequest.Error,
        )
        try:
            order_request = OrderRequest.from_dict(
                user=request.user,
                data=request.arguments,
                context=context,
            )
            result = self.order_manager.create_order(order_request)
        except handled_error_types as e:
            return Response({
                'status': 'errors',
                'errors': [str(e)],
                'server_time': time.time(),
            })

        order = result.order
        order.prefetch_related_objects()
        order_data = OrderSerializer(order).data

        data = {
            'status': 'success',
            'message': result.message.to_dict() if result.message else None,
            'server_time': time.time(),
            'order': order_data,
        }

        return Response(data)

    def get_queryset(self):
        return (
            Order.objects
            .with_related()
            .using(cars.settings.DB_RO_ID)
            .filter(user=self.request.user)
            .order_by('-created_at')
        )

    def get_reqans_logger_policy(self, request):
        if request.method == 'POST':
            policy = ReqAnsLogger.Policy.FULL
        else:
            policy = ReqAnsLogger.Policy.META
        return policy


class OrderItemDetailsView(RetrieveModelMixin, DriveAPIView):

    lookup_url_kwarg = 'order_item_id'
    serializer_class = OrderItemSerializer

    order_manager = OrderManager.from_settings(push_client=PUSH_CLIENT)

    def get_arguments_serializer_class(self):
        if self.request.method == 'POST':
            serializer_class = OrderItemDetailsArgumentsSerializer
        else:
            serializer_class = None
        return serializer_class

    def get_queryset(self):
        return (
            OrderItem.objects
            .with_related()
            .filter(
                id=self.kwargs['order_item_id'],
                order_id=self.kwargs['order_id'],
                order__user_id=self.request.user.id,
            )
        )

    def do_get(self, request, *args, **kwargs):
        return self.retrieve(request, *args, **kwargs)

    def do_post(self, request, order_id, order_item_id):
        try:
            order_item = (
                OrderItem.objects
                .get(
                    id=order_item_id,
                    order_id=order_id,
                    order__user_id=request.user.id,
                )
            )
        except OrderItem.DoesNotExist:
            raise NotFound

        context = OrderItemActionContext()

        error = None
        try:
            order = self.order_manager.send_action(
                order_item,
                action=request.arguments['action'],
                params=request.arguments.get('params'),
                context=context,
            )
        except BaseOrderItemManager.ActionError as e:
            error = '{}.{}'.format(order_item.type, e.code)
            order = None
        except BaseOrderItemManager.InvalidActionError:
            LOGGER.warning(
                'invalid action %s for order item %s',
                request.arguments['action'],
                order_item.id,
            )
            return Response(status=HTTP_400_BAD_REQUEST)

        if order is None:
            order = Order.objects.get(id=order_id)

        order.prefetch_related_objects()

        data = {
            'order': OrderSerializer(order).data,
            'server_time': time.time(),
        }
        if error:
            data['status'] = 'errors'
            data['errors'] = [error]
        else:
            data['status'] = 'success'

        return Response(data=data)

    def get_reqans_logger_policy(self, request):
        if request.method == 'POST':
            policy = ReqAnsLogger.Policy.FULL
        else:
            policy = ReqAnsLogger.Policy.META
        return policy


class OrderItemListView(DriveAPIView):

    arguments_serializer_class = OrderItemRequestSerializer

    order_manager = OrderManager.from_settings(push_client=PUSH_CLIENT)

    def do_post(self, request, order_id):
        try:
            order = (
                Order.objects
                .filter(
                    user=request.user,
                    completed_at__isnull=True,
                )
                .get(id=order_id)
            )
        except Order.DoesNotExist:
            raise NotFound

        context = OrderItemRequestContext(
            oauth_token=request.oauth_token,
        )

        order_item_request = OrderItemRequest.from_dict(
            user=request.user,
            data=request.arguments,
            context=context,
        )

        try:
            result = self.order_manager.add_order_item(
                order=order,
                order_item_request=order_item_request,
            )
        except OrderItemRequest.Error as e:
            return Response({
                'status': 'errors',
                'errors': [str(e)],
            })

        result.order.prefetch_related_objects()
        data = {
            'status': 'success',
            'message': result.message.to_dict() if result.message else None,
            'order': OrderSerializer(result.order).data,
            'server_time': time.time(),
        }

        return Response(data)

    def get_reqans_logger_policy(self, request):
        if request.method == 'POST':
            policy = ReqAnsLogger.Policy.FULL
        else:
            policy = ReqAnsLogger.Policy.META
        return policy
