from django.shortcuts import get_object_or_404
from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST

import cars.settings
from cars.core.mds.wrapper import MDSDocumentsWrapper
from cars.core.images import ImageProcessor, Resizer
from cars.django.parsers import ImageBinaryStreamParser, VideoBinaryStreamParser
from cars.django.renderers import JPEGRenderer
from cars.django.util import parse_uuid_or_404
from cars.users.core import UserProfileUpdater
from cars.users.models import UserDocumentPhoto

from .base import DriveAPIView


class UserDocumentPhotoThumbnailView(DriveAPIView):

    lookup_url_kwarg = 'photo_id'
    renderer_classes = [JPEGRenderer]

    mds_client = MDSDocumentsWrapper.from_settings()

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        resizer = Resizer(
            cars.settings.DRIVE['document_images']['greater_side_to_fit']
        )

        self._image_processor = ImageProcessor(
            watermarker=None,
            resizer=resizer,
        )

    def get_queryset(self):
        return (
            UserDocumentPhoto.objects
            .filter(
                user=self.request.user,
                document_id=self.kwargs['document_id'],
            )
        )

    def do_get(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        user_document_photo = self.get_object()
        data = self.mds_client.get_user_document_photo(
            user_document_photo=user_document_photo,
            image_processor=self._image_processor,
        )
        return Response(data)


class UserDocumentBackgroundVideoView(DriveAPIView):

    parser_classes = [
        ImageBinaryStreamParser,
        VideoBinaryStreamParser,
    ]

    mds_client = MDSDocumentsWrapper.from_settings()

    def do_put(self, request, document_id, photo_id):
        if not request.data:
            return Response(status=HTTP_400_BAD_REQUEST)

        document_id = parse_uuid_or_404(document_id)
        photo_id = parse_uuid_or_404(photo_id)
        document_photo = get_object_or_404(
            UserDocumentPhoto,
            user=request.user,
            document_id=document_id,
            id=photo_id,
        )

        updater = UserProfileUpdater(
            user=request.user,
            mds_client=self.mds_client,
        )
        updater.upload_document_photo_background_video(
            document_photo=document_photo,
            video_stream=request.data,
            mime_type=request.content_type,
        )

        return Response()
