from rest_framework.response import Response

from cars.users.core import UserPhoneBinder
from ..serializers.user_phones import (
    CommitUserPhonesArgumentsSerializer, SubmitUserPhonesArgumentsSerializer,
)
from .base import DriveAPIView


class BaseUserPhonesView(DriveAPIView):

    _phone_binder = None

    @classmethod
    def _get_phone_binder(cls):
        if cls._phone_binder is None:
            cls._phone_binder = UserPhoneBinder.from_settings()
        return cls._phone_binder


class CommitUserPhonesView(BaseUserPhonesView):

    arguments_serializer_class = CommitUserPhonesArgumentsSerializer

    def do_post(self, request):
        phone_binder = self._get_phone_binder()

        errors = None
        try:
            phone_binder.bind_commit(
                user=request.user,
                user_ip=request.user_ip,
                user_ua=request.user_agent,
                code=request.arguments['code'],
                oauth_token=request.oauth_token,
            )
        except phone_binder.PassportError as e:
            errors = e.response.data['errors']
        except phone_binder.NotSubmittedError:
            errors = ['submission.missing']

        if errors is not None:
            response_data = {
                'status': 'errors',
                'errors': errors,
            }
            return Response(response_data)

        response_data = {
            'status': 'success',
        }
        return Response(response_data)


class SubmitUserPhonesView(BaseUserPhonesView):

    arguments_serializer_class = SubmitUserPhonesArgumentsSerializer

    def do_post(self, request):
        phone_binder = self._get_phone_binder()

        errors = None
        try:
            phone_binder.bind_submit(
                user=request.user,
                user_ip=request.user_ip,
                user_ua=request.user_agent,
                number=request.arguments['number'],
                display_language='ru',
                oauth_token=request.oauth_token,
            )
        except phone_binder.PassportError as e:
            errors = e.response.data['errors']
        except phone_binder.RecentSubmissionError:
            errors = ['limit.exceeded']

        if errors is not None:
            response_data = {
                'status': 'errors',
                'errors': errors,
            }
            return Response(response_data)

        response_data = {
            'status': 'success',
        }
        return Response(response_data)
