import io
import json
import logging
import time
from collections import defaultdict

from lxml import etree

from kubiki.util import make_requests_session

import cars.settings


LOGGER = logging.getLogger(__name__)


class EKAClient:

    ENDPOINT_URL = 'https://apizenda.eka.ru/ZendaPrivateOfficeService.asmx'
    HOST = 'apizenda.eka.ru'

    BLOCK_CARD_SOAP_ACTION = 'http://zenda/exchange/reports/BlockCard'
    GET_STATUS_SOAP_ACTION = 'http://zenda/exchange/reports/GetCardIsEnabled'
    SET_STATUS_SOAP_ACTION = 'http://zenda/exchange/reports/SetCardIsEnabled'
    GET_SERVICE_SOAP_ACTION = 'http://zenda/exchange/reports/GetCardService'
    UNBLOCK_WITH_LIMIT_SOAP_ACTION = 'http://zenda/exchange/reports/UnblockCardWithLimit'

    GET_CARD_STATUS_PAYLOAD = """<?xml version="1.0" encoding="utf-8"?>
    <soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">
        <soap:Body>
            <GetCardIsEnabled xmlns="http://zenda/exchange/reports/">
                <userName>{}</userName>
                <password>{}</password>
                <cardNumber>{}</cardNumber>
            </GetCardIsEnabled>
        </soap:Body>
    </soap:Envelope>"""

    SET_CARD_STATUS_PAYLOAD = """<?xml version="1.0" encoding="utf-8"?>
    <soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">
        <soap:Body>
            <SetCardIsEnabled xmlns="http://zenda/exchange/reports/">
                <userName>{}</userName>
                <password>{}</password>
                <cardNumber>{}</cardNumber>
                <isEnabled>{}</isEnabled>
            </SetCardIsEnabled>
        </soap:Body>
    </soap:Envelope>"""

    BLOCK_CARD_PAYLOAD = """<?xml version="1.0" encoding="utf-8"?>
    <soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">
        <soap:Body>
            <BlockCard xmlns="http://zenda/exchange/reports/">
                <userName>{}</userName>
                <password>{}</password>
                <cardNumber>{}</cardNumber>
                <nomenclature>{}</nomenclature>
            </BlockCard>
        </soap:Body>
    </soap:Envelope>"""

    UNBLOCK_WITH_LIMIT_PAYLOAD = """<?xml version="1.0" encoding="utf-8"?>
    <soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">
        <soap:Body>
            <UnblockCardWithLimit xmlns="http://zenda/exchange/reports/">
                <userName>{}</userName>
                <password>{}</password>
                <cardNumber>{}</cardNumber>
                <nomenclature>{}</nomenclature>
                <limit>60.0</limit>
            </UnblockCardWithLimit>
        </soap:Body>
    </soap:Envelope>"""

    GET_CARD_SERVICE_PAYLOAD = """<?xml version="1.0" encoding="utf-8"?>
    <soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">
      <soap:Body>
        <GetCardService xmlns="http://zenda/exchange/reports/">
          <userName>{}</userName>
          <password>{}</password>
          <dateFrom>{}</dateFrom>
          <dateTo>{}</dateTo>
          <settlementAccount>{}</settlementAccount>
        </GetCardService>
      </soap:Body>
    </soap:Envelope>"""

    XSLT_PATTERN = """<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
        <xsl:output method="xml" indent="no"/>
        <xsl:template match="/|comment()|processing-instruction()">
            <xsl:copy>
                <xsl:apply-templates/>
            </xsl:copy>
        </xsl:template>
        <xsl:template match="*">
            <xsl:element name="{local-name()}">
                <xsl:apply-templates select="@*|node()"/>
            </xsl:element>
        </xsl:template>
        <xsl:template match="@*">
            <xsl:attribute name="{local-name()}">
                <xsl:value-of select="."/>
            </xsl:attribute>
        </xsl:template>
    </xsl:stylesheet>"""

    NOMENCLATURE_TO_EKA = {
        '92': '001000092',
        '95': '001000095',
        '98': '001000098',
    }

    def __init__(self, login, password, settlement_account=None):
        self._login = login
        self._password = password
        self._settlement_account = settlement_account

        self._session = make_requests_session()

    @classmethod
    def from_settings(cls):
        return cls(
            login=cars.settings.EKA['login'],
            password=cars.settings.EKA['password'],
            settlement_account=cars.settings.EKA['settlement_account'],
        )

    def get_card_status(self, number):
        payload = self.GET_CARD_STATUS_PAYLOAD.format(self._login, self._password, number)
        response = self._session.post(
            self.ENDPOINT_URL,
            data=payload,
            verify=False,
            headers={
                'Content-Type': 'text/xml; charset=utf-8',
                'Content-Length': str(len(payload)),
                'SOAPAction': self.GET_STATUS_SOAP_ACTION,
                'Host': self.HOST,
            },
            timeout=100,
        )
        response.raise_for_status()

        xslt_doc = etree.parse(io.BytesIO(self.XSLT_PATTERN.encode('utf-8')))
        transform = etree.XSLT(xslt_doc)
        dom = transform(etree.parse(io.BytesIO(response.text.encode('utf-8'))))

        is_enabled_nodes = dom.xpath(
            '/Envelope/Body/GetCardIsEnabledResponse/GetCardIsEnabledResult/IsEnabled'
        )
        if not is_enabled_nodes:
            raise RuntimeError('no IsEnabled nodes')
        if len(is_enabled_nodes) > 1:
            raise RuntimeError('more that one IsEnabled node')

        if is_enabled_nodes[0].text == 'true':
            return True

        return False

    def set_card_status_old(self, number, is_active):
        time_before = time.time()

        if is_active:
            status = 'true'
        else:
            status = 'false'

        payload = self.SET_CARD_STATUS_PAYLOAD.format(self._login, self._password, number, status)
        response = self._session.post(
            self.ENDPOINT_URL,
            data=payload,
            verify=False,
            headers={
                'Content-Type': 'text/xml; charset=utf-8',
                'Content-Length': str(len(payload)),
                'SOAPAction': self.SET_STATUS_SOAP_ACTION,
                'Host': self.HOST,
            },
            timeout=100,
        )
        response.raise_for_status()

        total_time_spent = time.time() - time_before
        LOGGER.info('total request time for number %s is %s', str(number), str(total_time_spent))

    def unblock_with_limit(self, number, fuel_type):
        nomenclature = self.NOMENCLATURE_TO_EKA[fuel_type]
        time_before = time.time()

        payload = self.UNBLOCK_WITH_LIMIT_PAYLOAD.format(self._login, self._password, number, nomenclature)
        headers = {
            'Content-Type': 'text/xml; charset=utf-8',
            'Content-Length': str(len(payload)),
            'SOAPAction': self.UNBLOCK_WITH_LIMIT_SOAP_ACTION,
            'Host': self.HOST,
        }

        response = self._session.post(
            self.ENDPOINT_URL,
            data=payload,
            verify=False,
            headers=headers,
            timeout=100,
        )
        response.raise_for_status()

        total_time_spent = time.time() - time_before
        LOGGER.info('total EKA request time for number %s is %s', str(number), str(total_time_spent))
        LOGGER.info(
            'requesting EKA with payload %s and headers %s',
            payload.replace(self._password, '%PASSWORD%'),
            json.dumps(headers),
        )
        LOGGER.info('EKA response is %s', response.text)
        LOGGER.info('EKA request status code is %d', response.status_code)
        LOGGER.info('EKA fuel card status: %s', str(self.get_card_status(number)))

    def block_card(self, number, fuel_type):
        nomenclature = self.NOMENCLATURE_TO_EKA[fuel_type]
        time_before = time.time()

        payload = self.BLOCK_CARD_PAYLOAD.format(self._login, self._password, number, nomenclature)
        response = self._session.post(
            self.ENDPOINT_URL,
            data=payload,
            verify=False,
            headers={
                'Content-Type': 'text/xml; charset=utf-8',
                'Content-Length': str(len(payload)),
                'SOAPAction': self.BLOCK_CARD_SOAP_ACTION,
                'Host': self.HOST,
            },
            timeout=100,
        )
        response.raise_for_status()

        total_time_spent = time.time() - time_before
        LOGGER.info('total request time for number %s is %s', str(number), str(total_time_spent))

    def get_card_service(self, date_from, date_to):
        payload = self.GET_CARD_SERVICE_PAYLOAD.format(
            self._login,
            self._password,
            date_from.isoformat(),
            date_to.isoformat(),
            self._settlement_account,
        )

        response = self._session.post(
            self.ENDPOINT_URL,
            data=payload,
            verify=False,
            headers={
                'Content-Type': 'text/xml; charset=utf-8',
                'Content-Length': str(len(payload)),
                'SOAPAction': self.GET_SERVICE_SOAP_ACTION,
                'Host': self.HOST,
            },
            timeout=100,
        )


class EKAClientStub:

    def __init__(self, login=None, password=None, settlement_account=None):
        self._statuses = defaultdict(bool)

    @classmethod
    def from_settings(cls):
        return cls()

    def get_card_status(self, number):
        return self._statuses[number]

    def set_card_status_old(self, number, is_active):
        self._statuses[number] = is_active

    def block_card(self, number, fuel_type):  # pylint:disable=unused-arguments
        self._statuses[number] = False

    def unblock_with_limit(self, number, fuel_type):  # pylint:disable=unused-arguments
        self._statuses[number] = True
