import datetime
import logging
import threading

from django.db import transaction
from django.utils import timezone

import cars.settings
from cars.billing.core.bonus_account_manager import BonusAccountManager
from cars.carsharing.core.car_updater import CarUpdater
from cars.users.models import User
from .eka_client import EKAClient
from ..models import FuelCardActivation


LOGGER = logging.getLogger(__name__)


class FuelCardProcessor:

    eka_client = EKAClient.from_settings()

    def __init__(self, time_to_expiry, operator, source, order=None, car=None):
        self._order = order
        self._time_to_expiry = datetime.timedelta(seconds=time_to_expiry)

        self._operator = operator
        self._source = source

        if order is not None:
            self._car = order.items.first().get_impl().car
        else:
            self._car = car

    @classmethod
    def from_activation_record(cls, activation_record, operator, source):
        return cls(
            operator=operator,
            source=source,
            car=activation_record.car,
            order=activation_record.order,
            time_to_expiry=cars.settings.EKA['time_to_block_card'],
        )

    @classmethod
    def from_order(cls, order, operator, source):
        return cls(
            order=order,
            operator=operator,
            source=source,
            time_to_expiry=cars.settings.EKA['time_to_block_card'],
        )

    def activate_fuel_card(self):
        """
        Activate fuel card and return 4-digit PIN.
        """
        activation_record = (
            FuelCardActivation.objects
            .filter(
                order=self._order,
                car=self._car,
                blocked_at__isnull=True,
            )
            .first()
        )
        if activation_record is None:
            activation_record = FuelCardActivation(
                order=self._order,
                car=self._car,

                activated_at=timezone.now(),
                activated_by=self._operator,
                activated_via=self._source,

                initial_fuel_level=self._car.telematics_state.fuel_level,
                eventual_fuel_level=self._car.telematics_state.fuel_level,
            )
            activation_record.save()
        else:
            activation_record.activated_at = timezone.now()
            activation_record.initial_fuel_level = self._car.telematics_state.fuel_level
            activation_record.eventual_fuel_level = self._car.telematics_state.fuel_level
            activation_record.save()

        t = threading.Thread(
            target=self.eka_client.unblock_with_limit,
            kwargs={
                'number': self._car.fuel_card_number,
                'fuel_type': self._car.model.fuel_type,
            },
        )
        t.start()

        return {
            'pin': self._car.fuel_card_number[-4:],
            'time_remaining': (
                (activation_record.activated_at + self._time_to_expiry - timezone.now())
                .total_seconds()
            ),
        }

    def block_fuel_card(self):
        activation_record = (
            FuelCardActivation.objects
            .filter(
                order=self._order,
                car=self._car,
                blocked_at__isnull=True,
            )
            .first()
        )

        if activation_record is None:
            LOGGER.error(
                'no fuel card activation record for car=%s order=%s',
                str(self._car.id),
                str(self._order),
            )

        t = threading.Thread(
            target=self.eka_client.block_card,
            kwargs={
                'number': self._car.fuel_card_number,
                'fuel_type': self._car.model.fuel_type,
            },
        )
        t.start()

        if activation_record is not None:
            activation_record.blocked_at = timezone.now()
            activation_record.blocked_by = self._operator
            activation_record.blocked_via = self._source
            activation_record.save()

    def change_card_number(self, new_number):
        car_updater = CarUpdater(self._car)
        car_updater.update_fuel_card_number(new_number)


class ExpiredActivationsBlocker:

    def __init__(self, time_to_expiry, robot_user_id):
        self._time_to_expiry = datetime.timedelta(seconds=time_to_expiry)
        self._robot_user_id = robot_user_id

    @classmethod
    def from_settings(cls):
        return cls(
            time_to_expiry=cars.settings.EKA['time_to_block_card'],
            robot_user_id=cars.settings.EKA['robot_user_id'],
        )

    def block_expired_fuel_cards(self):
        target_activations = (
            FuelCardActivation.objects
            .filter(
                activated_at__lte=timezone.now() - self._time_to_expiry,
                blocked_at__isnull=True,
            )
        )

        n_blocked = 0
        operator = User.objects.get(id=self._robot_user_id)

        for activation in target_activations:
            processor = FuelCardProcessor.from_activation_record(
                activation_record=activation,
                operator=operator,
                source=FuelCardActivation.Source.ROBOT_AUTO.value,
            )
            try:
                processor.block_fuel_card()
                n_blocked += 1
            except Exception:
                LOGGER.error(
                    'unable to deactivate fuel card activation_id=%s order_id=%s',
                    str(activation.id),
                    str(activation.order.id)
                )

        return n_blocked


class FuelLevelVerifier:

    def __init__(self, time_to_scan, consider_fueled_threshold):
        self._time_to_scan = datetime.timedelta(seconds=time_to_scan)
        self._consider_fueled_threshold = consider_fueled_threshold

    @classmethod
    def from_settings(cls):
        return cls(
            time_to_scan=cars.settings.EKA['time_to_scan_fuel'],
            consider_fueled_threshold=cars.settings.EKA['consider_fueled_threshold'],
        )

    def scan_fuel_level_from_recently_fueled_cars(self):
        recent_unblockings = (
            FuelCardActivation.objects
            .filter(
                order__isnull=False,
                blocked_at__gte=timezone.now() - self._time_to_scan,
            )
            .select_related('car__telematics_state')
            .order_by('-blocked_at')
        )

        n_just_fueled = 0
        processed_order_ids = set()
        for u in recent_unblockings:
            if u.order_id in processed_order_ids:
                continue
            processed_order_ids.add(u.order_id)
            if u.eventual_fuel_level >= self._consider_fueled_threshold:
                continue

            if u.car is None:
                # some of the first fueling records did not contain car entry, fix that
                u.car = u.order.items.first().get_impl().car
                u.save()

            if u.car.telematics_state.fuel_level >= self._consider_fueled_threshold:
                u.eventual_fuel_level = u.car.telematics_state.fuel_level
                u.save()

                n_just_fueled += 1

        return n_just_fueled


class BonusAccountUpdater:

    def __init__(self, consider_fueled_threshold, bonus_amount, bonus_comment):
        self._consider_fueled_threshold = consider_fueled_threshold
        self._bonus_amount = bonus_amount
        self._bonus_comment = bonus_comment

    @classmethod
    def from_settings(cls):
        return cls(
            consider_fueled_threshold=cars.settings.EKA['consider_fueled_threshold'],
            bonus_amount=cars.settings.EKA['bonus_amount'],
            bonus_comment=cars.settings.EKA['bonus_comment'],
        )

    def assign_bonuses_for_users(self):
        activations_for_bonus = (
            FuelCardActivation.objects
            .filter(
                order__isnull=False,
                eventual_fuel_level__gte=self._consider_fueled_threshold,
                bonus_account_operation__isnull=True,
            )
            .select_related('order__user')
        )

        orders_with_bonus = self._get_orders_with_bonus_among_qs(activations_for_bonus)

        n_updated = 0
        for activation in activations_for_bonus:
            if str(activation.order_id) in orders_with_bonus:
                # don't refund twice
                continue

            with transaction.atomic():
                manager = BonusAccountManager.from_user(activation.order.user)
                account = manager.debit_generic(
                    amount=self._bonus_amount,
                    operator=activation.order.user,
                    comment=self._bonus_comment,
                    nonce=str(activation.id),
                )
                activation.bonus_account_operation = (
                    account.operations
                    .filter(nonce=str(activation.id))
                    .first()
                )
                activation.save()

                orders_with_bonus.add(str(activation.order_id))

            n_updated += 1

        return n_updated

    def _get_orders_with_bonus_among_qs(self, qs):
        order_ids = []
        for entry in qs:
            order_ids.append(entry.order_id)

        activations_with_bonus = (
            FuelCardActivation.objects
            .filter(
                order_id__in=order_ids,
                bonus_account_operation__isnull=False,
            )
        )

        orders_with_bonus = set()
        for activation in activations_with_bonus:
            orders_with_bonus.add(str(activation.order_id))

        return orders_with_bonus
