from rest_framework.response import Response

import cars.settings
from cars.carsharing.core.telematics_proxy import TelematicsApiResponse
from cars.core.util import import_class
from cars.django.permissions import IsAuthenticated
from cars.django.views import CarsharingAPIView
from cars.drive.serializers.order import OrderSerializer
from cars.orders.models import OrderItem


class CarTelematicsActionBaseView(CarsharingAPIView):

    permission_classes = [IsAuthenticated]
    telematics_proxy = import_class(cars.settings.TELEMATICS['proxy_class']).from_settings()

    def post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        order_item = (
            OrderItem.objects
            .with_related()
            .filter(
                order__user_id=request.user.id,
                finished_at__isnull=True,
            )
            .select_related('order')
            .first()
        )

        if order_item is None:
            return Response(
                {
                    'status': 'errors',
                    'errors': ['order.missing'],
                    'order': None,
                },
            )

        car = order_item.get_impl().car

        status = 'success'
        error = None
        try:
            response = self._do_car_command(car)
            response.raise_for_status()
        except TelematicsApiResponse.Error as e:
            assert e.code is not None
            status = 'errors'
            error = '{}.car.{}'.format(order_item.type, e.code)

        order = order_item.order
        order.refresh_from_db()
        order.prefetch_related_objects()

        response_dict = {
            'status': status,
            'order': OrderSerializer(order).data,
        }
        if status == 'errors':
            response_dict['errors'] = [error]

        return Response(
            response_dict
        )

    def _do_car_command(self, car):
        raise NotImplementedError


class CarOpenDoorsView(CarTelematicsActionBaseView):

    def _do_car_command(self, car):
        return self.telematics_proxy.open(car.imei)


class CarCloseDoorsView(CarTelematicsActionBaseView):

    def _do_car_command(self, car):
        """
        if car.telematics_state.front_left_door_open:
            raise TelematicsApiResponse.Error(code='driver_door_open')
        if car.telematics_state.front_right_door_open:
            raise TelematicsApiResponse.Error(code='pass_door_open')
        if car.telematics_state.rear_left_door_open:
            raise TelematicsApiResponse.Error(code='left_rear_door_open')
        if car.telematics_state.rear_right_door_open:
            raise TelematicsApiResponse.Error(code='right_rear_door_open')
        if car.telematics_state.trunk_open:
            raise TelematicsApiResponse.Error(code='trunk_door_open')
        """

        return self.telematics_proxy.close(car.imei)
