from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST

from cars.django.permissions import IsAuthenticated
from cars.django.views import CarsharingAPIView
from cars.eka.models import FuelCardActivation
from cars.orders.models import Order
from ..core.processor import FuelCardProcessor
from ..serializers.fuel_card import FuelCardNumberUpdateArgumentsSerializer


class FuelCardActionBaseView(CarsharingAPIView):

    permission_classes = [IsAuthenticated]

    def __init__(self, *args, **kwargs):
        self.processor = None
        super().__init__(*args, **kwargs)

    def post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        order = (
            Order.objects
            .filter(
                user_id=request.user.id,
                completed_at__isnull=True,
            )
            .first()
        )

        if order is None:
            return Response(
                {
                    'error': 'order.missing',
                },
                status=HTTP_400_BAD_REQUEST,
            )

        self.processor = FuelCardProcessor.from_order(
            order=order,
            operator=request.user,
            source=FuelCardActivation.Source.MOBILE_APP.value,
        )

        return super().post(request, *args, **kwargs)


class FuelCardActivateView(FuelCardActionBaseView):

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        return Response(
            self.processor.activate_fuel_card(),
        )


class FuelCardBlockView(FuelCardActionBaseView):

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        self.processor.block_fuel_card()
        return Response()


class FuelCardNumberUpdateView(FuelCardActionBaseView):

    arguments_serializer_class = FuelCardNumberUpdateArgumentsSerializer

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        self.processor.change_card_number(request.arguments['number'])
        return Response()
