# -*- coding: utf-8 -*-


import logging
from requests import get
from requests.exceptions import RequestException
from json import loads


LOGGER = logging.getLogger(__name__)


class FrontendApiClient(object):
    BASE_URL = 'http://admin.carsharing.yandex.net/api'

    TRY_COUNT = 5

    BUILD_VERSION = 100500  # kostyl

    def __init__(self, token):
        self._token = token

    def get_session_raw_route(self, session):
        data = self._load_data('/staff/track/trace', {'session': session})
        route = []
        tracks = data.get('tracks')
        if not isinstance(tracks, list):
            return route
        for track in tracks:
            timestamps = track.get('timestamps', '').split(' ')
            coords = track.get('coords', '').split(' ')
            # Last coordinate may be empty, so remove it
            if coords[-1] == '':
                coords.pop()
            if len(timestamps) * 2 != len(coords):
                LOGGER.error(
                    'For each point should be only one timestamp!'
                    '{} * 2 != {}'
                    .format(len(timestamps), len(coords))
                )
            for i, timestamp in enumerate(timestamps):
                route.append((
                    float(coords[i * 2 + 1]),  # Latitude
                    float(coords[i * 2]),  # Longitude
                    int(timestamp)
                ))
        # Points in route sometimes are not sorted
        route.sort(key=lambda point: point[2])
        return route

    def get_session_route(self, session):
        data = self._load_data('/staff/track/analyzer', {'session': session})
        route = []
        tracks = data.get('tracks')
        if not isinstance(tracks, list):
            return route
        tracks.sort(key=lambda track: track.get('start_timestamp', 0))
        for track in tracks:
            points = track.get('track', [])
            if len(points) % 2 != 0:
                LOGGER.error('Number of coordinates should be even')
            for i in range(len(points) // 2):
                route.append((
                    float(points[i * 2 + 1]),
                    float(points[i * 2])
                ))
        return route

    def _load_data(self, url, args={}):
        headers = {
            'Content-Type': 'application/json',
            'Authorization': 'OAuth {}'.format(self._token),
        }
        for try_number in range(self.TRY_COUNT):
            try:
                response = get(
                    self.BASE_URL + url,
                    headers=headers,
                    params=args,
                    verify=False,
                )
                if response.status_code == 200:
                    return loads(response.content.decode('utf-8'))
            except RequestException as exc:
                LOGGER.exception(exc)
        raise RuntimeError()
