# -*- coding: utf-8 -*-

from logging import getLogger
from datetime import datetime, timedelta
from collections import defaultdict
from yt.wrapper import with_context
from cars import settings
from .cron_job import YtCronJob
from cars.export.tables.cars import CarTagsHistory
from cars.export.tables.orders import Orders, Routes, CarTagsQueue


LOGGER = getLogger(__name__)


class OrdersExportJob(YtCronJob):
    code = 'orders_export'

    tick_interval = '0 2,4,6,12,20 * * *'

    FIRST_STATE = '2018-01-01'

    def _do_tick(self):
        state = (datetime.utcnow() - timedelta(hours=3)).replace(
            hour=0,
            minute=0,
            second=0,
            microsecond=0,
        )
        last_state = self._get_last_state()
        if state > last_state:
            while last_state < state:
                self._set_last_state(last_state)
                self._update(last_state, True)
                last_state += timedelta(days=1)
        self._set_last_state(state)
        self._update(state, False)

    def _get_last_state(self):
        last_state = self._yt.get_attribute(
            '{}/{}'.format(
                settings.EXPORT['home_directory'],
                Orders.ORDERS_DIR
            ),
            '_last_state',
            self.FIRST_STATE
        )
        return (
            datetime.strptime(last_state, '%Y-%m-%d')
            .replace(
                hour=0,
                minute=0,
                second=0,
                microsecond=0,
            )
        )

    def _set_last_state(self, state):
        self._yt.set_attribute(
            '{}/{}'.format(
                settings.EXPORT['home_directory'],
                Orders.ORDERS_DIR
            ),
            '_last_state',
            state.strftime("%Y-%m-%d")
        )

    def _update(self, day, complete):
        orders = Orders(day)
        routes = Routes(day)
        tags_queue = CarTagsQueue()
        tags_history = CarTagsHistory(day)
        routes.save(self._yt, complete)
        with self._yt.Transaction(timeout=self.transaction_timeout):
            orders.save(self._yt)
            if self._yt.exists(tags_history._get_path()):
                tags_queue.push_table(tags_history, self._yt)
            if complete:
                tags_history._day += timedelta(days=1)
                tags_queue.push_table(tags_history, self._yt)
            if self._yt.exists(tags_queue._get_path()):
                self._improve_orders(orders, tags_queue, complete)
            self._merge_orders_and_routes(orders, routes)

    def _improve_orders(self, orders, tags_queue, clear_queue):
        with self._yt.TempTable(settings.EXPORT['temp_path']) as orders_table:
            self._yt.run_sort(
                source_table=orders._get_path(),
                destination_table=orders_table,
                sort_by=['car_id', 'created_at'],
            )
            with self._yt.TempTable(settings.EXPORT['temp_path']) as tags_table:
                self._yt.run_sort(
                    source_table=tags_queue._get_path(),
                    destination_table=tags_table,
                    sort_by=['car_id', 'event_id'],
                )
                self._yt.run_reduce(
                    self._orders_improver,
                    source_table=[
                        orders_table,
                        tags_table,
                    ],
                    destination_table=orders._get_path(),
                    reduce_by=['car_id'],
                )
                if clear_queue:
                    self._yt.run_reduce(
                        self._queue_cleaner,
                        source_table=[
                            orders_table,
                            tags_table,
                        ],
                        destination_table=tags_queue._get_path(),
                        reduce_by=['car_id'],
                    )

    def _merge_orders_and_routes(self, orders, routes):
        with self._yt.TempTable(settings.EXPORT['temp_path']) as orders_table:
            self._yt.run_sort(
                source_table=orders._get_path(),
                destination_table=orders_table,
                sort_by=['session_id'],
            )
            with self._yt.TempTable(settings.EXPORT['temp_path']) as routes_table:
                self._yt.run_sort(
                    source_table=routes._get_path(),
                    destination_table=routes_table,
                    sort_by=['session_id'],
                )
                self._yt.run_reduce(
                    self._orders_routes_merger,
                    source_table=[orders_table, routes_table],
                    destination_table=orders._get_path(),
                    reduce_by=['session_id'],
                )

    @staticmethod
    def _compile_order(order, tags):
        order['geo_id'] = 'msc_area'
        order['from_scanner'] = False
        order['overrun'] = 0.0
        order['overtime'] = 0
        order['duration_by_steps'] = []
        order['cost_by_steps'] = []
        prev_type, prev_timestamp = None, None
        riding_price = 0.0
        parking_price = 0.0
        riding_free_time = 0
        parking_free_time = 0
        reservation_free_time = 0
        acceptance_free_time = 0
        pricing_by_seconds = True
        mileage_limit = 0
        for tag in tags:
            if tag['tag'] == 'old_state_reservation':
                if tag['data'] and 'offer' in tag['data']:
                    offer = tag['data']['offer']
                    order['from_scanner'] = (
                        offer.get('from_scanner', False)
                        or offer.get('FromScanner', False)
                    )
                    order['tariff_name'] = offer.get('name') or offer.get('Name')
                    order['tariff_price'] = {}
                    if 'StandartOffer' in offer:
                        std_offer = offer['StandartOffer']
                        riding_price = std_offer.get('PriceRiding', 0) / 100
                        parking_price = std_offer.get('PriceParking', 0) / 100
                        order['tariff_price']['prices'] = {}
                        order['tariff_price']['prices']['riding'] = riding_price
                        order['tariff_price']['prices']['parking'] = parking_price
                    if 'data' in offer:
                        data = offer['data']
                        if 'mileage_limit' in data:
                            order['overrun'] = float(max(
                                0.0,
                                order['mileage'] - data['mileage_limit']
                            ))
                            order['tariff_price']['mileage_limit'] = data['mileage_limit']
                            mileage_limit = data['mileage_limit']
                        if 'pack_duration' in data:
                            duration = order['completed_at'] - order['created_at']
                            order['overtime'] = int(max(
                                0,
                                order['priced_duration'] - data['pack_duration']
                            ))
                            order['tariff_price']['pack_duration'] = data['pack_duration']
                        if 'prices' in data:
                            order['tariff_price']['prices'] = {}
                            for key, value in data['prices'].items():
                                order['tariff_price']['prices'][key] = value / 100
                                if key == 'riding':
                                    riding_price = value / 100
                                elif key == 'parking':
                                    parking_price = value / 100
                        if 'pack_price' in data:
                            order['tariff_price']['pack_price'] = data['pack_price'] / 100
                        if 'rerun_price_km' in data:
                            order['tariff_price']['rerun_price_km'] = data['rerun_price_km'] / 100
                        if 'pricing_by_seconds' in data:
                            order['tariff_price']['pricing_by_seconds'] = data['pricing_by_seconds']
                            pricing_by_seconds = data['pricing_by_seconds']
                        if 'discounts' in data:
                            for discount in data['discounts']:
                                for detail in discount.get('details', []):
                                    if detail['tag_name'] == 'old_state_riding':
                                        riding_free_time += detail.get('additional_duration', 0)
                                    elif detail['tag_name'] == 'old_state_parking':
                                        parking_free_time += detail.get('additional_duration', 0)
                                    elif detail['tag_name'] == 'old_state_reservation':
                                        reservation_free_time += detail.get('additional_duration', 0)
                                    elif detail['tag_name'] == 'old_state_acceptance':
                                        acceptance_free_time += detail.get('additional_duration', 0)
                if tag['snapshot'] and 'data' in tag['snapshot']:
                    data = tag['snapshot']['data']
                    if 'ptags' in data and isinstance(data['ptags'], list):
                        for ptag in data['ptags']:
                            if ptag == 'spb_area':
                                order['geo_id'] = 'spb_area'
                            if ptag == 'msc_area':
                                order['geo_id'] = 'msc_area'
                if tag['action'] == 'set_performer':
                    prev_type = 'old_state_reservation'
                    prev_timestamp = tag['timestamp']
                elif prev_type != tag['tag']:
                    order['duration_by_steps'].append({
                        'type': prev_type,
                        'begin': prev_timestamp,
                        'end': tag['timestamp'],
                    })
                    duration = tag['timestamp'] - prev_timestamp
                    step_cost = 0.
                    if prev_type == 'old_state_riding':
                        if pricing_by_seconds and not mileage_limit:
                            duration = max(0, duration - riding_free_time)
                            step_cost = duration / 60 * riding_price
                    elif prev_type == 'old_state_parking':
                        duration = max(0, duration - parking_free_time)
                        step_cost = duration / 60 * parking_price
                    elif prev_type == 'old_state_acceptance':
                        duration = max(0, duration - acceptance_free_time)
                        step_cost = duration / 60 * parking_price
                    elif prev_type == 'old_state_reservation':
                        duration = max(0, duration - reservation_free_time)
                        step_cost = duration / 60 * parking_price
                    order['cost_by_steps'].append({
                        'type': prev_type,
                        'cost': round(step_cost, 3),
                    })
                    prev_type = 'old_state_reservation'
                    prev_timestamp = tag['timestamp']
            elif tag['tag'] == 'old_state_acceptance' or \
                tag['tag'] == 'old_state_riding' or \
                tag['tag'] == 'old_state_parking':
                if prev_type != tag['tag']:
                    order['duration_by_steps'].append({
                        'type': prev_type,
                        'begin': prev_timestamp,
                        'end': tag['timestamp'],
                    })
                    duration = tag['timestamp'] - prev_timestamp
                    step_cost = 0.
                    if prev_type == 'old_state_riding':
                        if pricing_by_seconds and not mileage_limit:
                            duration = max(0, duration - riding_free_time)
                            step_cost = duration / 60 * riding_price
                    elif prev_type == 'old_state_parking':
                        duration = max(0, duration - parking_free_time)
                        step_cost = duration / 60 * parking_price
                    elif prev_type == 'old_state_acceptance':
                        duration = max(0, duration - acceptance_free_time)
                        step_cost = duration / 60 * parking_price
                    elif prev_type == 'old_state_reservation':
                        duration = max(0, duration - reservation_free_time)
                        step_cost = duration / 60 * parking_price
                    order['cost_by_steps'].append({
                        'type': prev_type,
                        'cost': round(step_cost, 3),
                    })
                    prev_type = tag['tag']
                    prev_timestamp = tag['timestamp']
        return order

    @staticmethod
    def _remove_invalid_tags(tags):
        result = []
        started = False
        for tag in tags:
            if not started and \
                (tag['tag'] != 'old_state_reservation' or \
                tag['action'] != 'set_performer'):
                continue
            started = True
            result.append(tag)
            if tag['action'] == 'drop_performer':
                break
        return result

    @staticmethod
    @with_context
    def _orders_improver(key, rows, context):
        orders = defaultdict(list)
        order_tags = defaultdict(list)
        it = defaultdict(int)
        tags = []
        for row in rows:
            if context.table_index == 0:
                orders[row['car_id']].append(row)
            else:
                tags.append(row)
        for tag in tags:
            car_id = tag['car_id']
            while it[car_id] < len(orders[car_id]) and \
                tag['timestamp'] > orders[car_id][it[car_id]]['completed_at']:
                it[car_id] += 1
            if it[car_id] == len(orders[car_id]) or \
                tag['timestamp'] < orders[car_id][it[car_id]]['created_at']:
                continue
            order = orders[car_id][it[car_id]]
            order_tags[order['session_id']].append(tag)
        for order_list in orders.values():
            for order in order_list:
                yield OrdersExportJob._compile_order(
                    order,
                    OrdersExportJob._remove_invalid_tags(
                        order_tags[order['session_id']]
                    )
                )

    @staticmethod
    @with_context
    def _orders_routes_merger(key, rows, context):
        orders = []
        routes = {}
        for row in rows:
            if context.table_index == 0:
                orders.append(row)
            else:
                routes[row['session_id']] = row
        for order in orders:
            session_id = order['session_id']
            if session_id in routes:
                order.update(routes[session_id])
            yield order

    @staticmethod
    @with_context
    def _queue_cleaner(key, rows, context):
        orders = defaultdict(list)
        it = defaultdict(int)
        tags = []
        for row in rows:
            if context.table_index == 0:
                orders[row['car_id']].append(row)
            else:
                tags.append(row)
        for tag in tags:
            car_id = tag['car_id']
            while it[car_id] < len(orders[car_id]) and \
                tag['timestamp'] > orders[car_id][it[car_id]]['completed_at']:
                it[car_id] += 1
            if it[car_id] == len(orders[car_id]) or \
                tag['timestamp'] < orders[car_id][it[car_id]]['created_at']:
                yield tag
