# -*- coding: utf-8 -*-

from logging import getLogger
from datetime import datetime, timedelta
from collections import defaultdict
from yt.wrapper import with_context
from cars import settings
from .cron_job import YtCronJob
from cars.export.tables.cars import CarTagsHistory
from cars.export.tables.orders import Orders, Routes, CarTagsQueue
from .orders_export import OrdersExportJob


LOGGER = getLogger(__name__)


class OrdersExportRecalcJob(OrdersExportJob):
    code = 'orders_export_recalc'

    tick_interval = '*/5 * * * *'

    def _do_tick(self):
        state = (datetime.utcnow() - timedelta(hours=3)).replace(
            hour=0,
            minute=0,
            second=0,
            microsecond=0,
        )
        last_state = self._get_last_state()
        if not last_state:
            return
        before_state = last_state - timedelta(days=5)
        while before_state < last_state:
            tags_queue = CarTagsQueue(recalc=True)
            tags_history = CarTagsHistory(before_state)
            if self._yt.exists(tags_history._get_path()):
                tags_queue.push_table(tags_history, self._yt)
            before_state += timedelta(days=1)
        if state > last_state:
            while last_state < state:
                self._set_last_state(last_state)
                self._update(last_state, True)
                last_state += timedelta(days=1)
        self._set_last_state(None)

    def _get_last_state(self):
        last_state = self._yt.get_attribute(
            '{}/{}'.format(
                settings.EXPORT['home_directory'],
                Orders.ORDERS_DIR
            ),
            '_last_state_recalc',
            None
        )
        if not last_state:
            return last_state
        return (
            datetime.strptime(last_state, '%Y-%m-%d')
            .replace(
                hour=0,
                minute=0,
                second=0,
                microsecond=0,
            )
        )

    def _set_last_state(self, state):
        self._yt.set_attribute(
            '{}/{}'.format(
                settings.EXPORT['home_directory'],
                Orders.ORDERS_DIR
            ),
            '_last_state_recalc',
            state.strftime("%Y-%m-%d") if state else None
        )

    def _update(self, day, complete):
        orders = Orders(day)
        routes = Routes(day)
        tags_queue = CarTagsQueue(recalc=True)
        tags_history = CarTagsHistory(day)
        with self._yt.Transaction(timeout=self.transaction_timeout):
            orders.save(self._yt)
            if self._yt.exists(tags_history._get_path()):
                tags_queue.push_table(tags_history, self._yt)
            if complete:
                tags_history._day += timedelta(days=1)
                tags_queue.push_table(tags_history, self._yt)
            if self._yt.exists(tags_queue._get_path()):
                self._improve_orders(orders, tags_queue, complete)
            if self._yt.exists(routes._get_path()):
                self._merge_orders_and_routes(orders, routes)
