# -*- coding: utf-8 -*-

import os
import logging
import dateutil.parser
from datetime import datetime, timedelta
from dateutil.relativedelta import relativedelta
from collections import defaultdict
from yt.wrapper import TablePath, with_context
from uuid import UUID
from cars import settings
from .cron_job import YtCronJob
from ..exports import (
    UsersExport, BonusAccountExport, DataSyncExport, OrdersExport, SecurityExport,
    UserProfilesExport, AutocodeFineExport, BillingTasksExport,
)
from ..tables import LicenseLossTable, PlusTable, BinAttrTable
from ..tables.users import UserTags

# Yt should not fail now
if os.environ.get('DJANGO_SETTINGS_MODULE') is not None:
    from cars.users.models import User, UserRole, AppInstall
    from cars.registration.models import RegistrationChatActionResult


LOGGER = logging.getLogger(__name__)


def _calc_age(birth_date):
    if not birth_date:
        return None
    birth_date = dateutil.parser.parse(birth_date).replace(tzinfo=None)
    try:
        delta = relativedelta(datetime.utcnow(), birth_date)
        return delta.years
    except:
        return None


def _calc_experience(start_date):
    if not start_date:
        return None
    start_date = dateutil.parser.parse(start_date).replace(tzinfo=None)
    try:
        delta = relativedelta(datetime.utcnow(), start_date)
        return delta.years
    except:
        return None


def _calc_gender(gender):
    if not gender:
        return None
    elif gender.lower() == 'муж':
        return 'm'
    elif gender.lower() == 'жен':
        return 'f'
    return None


def _calc_datasync_timestamp(date):
    if not date:
        return None
    try:
        return int(
            dateutil.parser
            .parse(date)
            .replace(tzinfo=None)
            .timestamp()
        )
    except ValueError:
        return None


def _users_apps_mapper(row):
    if row['DeviceID']:
        try:
            yield {
                'device_id': str(UUID(row['DeviceID'].lower())),
                'app_list': row['apps'],
                '_updated': int(row['EventTimestamp']),
            }
        except ValueError:
            pass


def _users_crypta_mapper(row):
    if row['id']:
        try:
            yield {
                'device_id': str(UUID(row['id'].lower())),
                'crypta_id': row['target_id'],
            }
        except ValueError:
            pass


@with_context
def _users_apps_reducer(key, rows, context):
    result_rows = []
    device_apps = {}
    for row in rows:
        if context.table_index == 0:
            result_rows.append({
                'id': row['id'],
                'device_id': row['device_id'],
            })
        else:
            device_id = row['device_id']
            if device_id not in device_apps:
                device_apps[device_id] = row
            elif device_apps[device_id]['_updated'] < row['_updated']:
                device_apps[device_id] = row
    for row in result_rows:
        device_id = row['device_id']
        if device_id in device_apps:
            yield {
                'id': row['id'],
                'app_list': device_apps[device_id]['app_list'],
                '_updated': device_apps[device_id]['_updated'],
            }


def _users_apps_merger(key, rows):
    result = {}
    for row in rows:
        user_id = row['id']
        if user_id not in result:
            result[user_id] = row
        elif result[user_id]['_updated'] < row['_updated']:
            result[user_id] = row
    for row in result.values():
        yield row


@with_context
def _users_crypta_reducer(key, rows, context):
    result_rows = []
    crypta_ids = defaultdict(lambda: None)
    for row in rows:
        if context.table_index == 0:
            result_rows.append({
                'id': row['id'],
                'device_id': row['device_id'],
            })
        else:
            crypta_ids[row['device_id']] = row['crypta_id']
    for row in result_rows:
        # Note, that we replaced id with user_id
        # and crypta_id with id
        # This simplify our reduce operations
        yield {
            'user_id': row['id'],
            'id': crypta_ids[row['device_id']],
        }


@with_context
def _users_crypta_merger(key, rows, context):
    result = []
    data = defaultdict(lambda: {'gaid': None, 'yandexuid': []})
    for row in rows:
        if context.table_index == 0:
            result.append(row)
        else:
            data_id = row['id']
            if context.table_index == 1:
                data[data_id]['gaid'] = row['target_id']
            elif context.table_index == 2:
                data[data_id]['yandexuid'].append(int(row['target_id']))
    for row in result:
        row.update(data[row['id']])
        # Now we need our normal format
        # id --- user uuid4
        # crypta_id --- crypta_id
        row['crypta_id'] = row['id']
        row['id'] = row['user_id']
        del row['user_id']
        if isinstance(row['crypta_id'], str):
            row['crypta_id'] = int(row['crypta_id'])
        yield row


def _users_crypta_profiles_mapper(row):
    for yandexuid in row['yandexuid']:
        yield {
            'id': row['id'],
            'yandexuid': yandexuid,
        }


@with_context
def _users_crypta_profiles_merger(key, rows, context):
    result = defaultdict(lambda: [])
    profiles = {}
    for row in rows:
        yandexuid = row['yandexuid']
        if context.table_index == 0:
            result[row['id']].append(yandexuid)
        else:
            if yandexuid not in profiles:
                profiles[yandexuid] = dict(row)
            elif row['update_time'] > profiles[yandexuid]['update_time']:
                profiles[yandexuid] = dict(row)
    for user_id, yandexuids in result.items():
        profile = None
        for yandexuid in yandexuids:
            if yandexuid not in profiles:
                continue
            if profile is None:
                profile = profiles[yandexuid]
            elif profile['update_time'] < profiles[yandexuid]['update_time']:
                profile = profiles[yandexuid]
        if yandexuid in profiles:
            profile = profiles[yandexuid]
            yield {
                'id': user_id,
                'crypta_profile': profile,
            }


def _large_table_merger(key, rows):
    result = defaultdict(lambda: {})
    for row in rows:
        row_id = row['id']
        data = {
            key: value
            for key, value in row.items()
                if not key.startswith('_') and value is not None
        }
        result[row_id].update(data)
    for row in result.values():
        if 'registered_at' in row and row['registered_at']:
            if row['registered_at'] <= '2000-01-01T00:00:00':
                row['registered_at'] = None
        row['age'] = _calc_age(row.get('birth_date_pass'))
        row['experience'] = _calc_experience(row.get('experience_from'))
        row['gender'] = _calc_gender(row.get('gender'))
        row['birth_date_dl'] = _calc_datasync_timestamp(row.get('birth_date_dl'))
        row['birth_date_pass'] = _calc_datasync_timestamp(row.get('birth_date_pass'))
        row['issue_date'] = _calc_datasync_timestamp(row.get('issue_date'))
        row['experience_from'] = _calc_datasync_timestamp(row.get('experience_from'))
        row['valid_to_date'] = _calc_datasync_timestamp(row.get('valid_to_date'))
        row['expiration_date'] = _calc_datasync_timestamp(row.get('expiration_date'))
        row['registration_expiration_date'] = _calc_datasync_timestamp(row.get('registration_expiration_date'))
        row['user_tags'] = row.get('user_tags', [])
        yield row


@with_context
def _current_score_reducer(key, rows, context):
    result_rows = []
    current_scores = defaultdict(float)
    for row in rows:
        if context.table_index == 0:
            result_rows.append(row['id'])
        else:
            current_scores[row['id']] = float(row['proba'])
    for user_id in result_rows:
        yield {
            'id': user_id,
            'current_score': current_scores[user_id],
        }


@with_context
def _plus_subscribers_reducer(key, rows, context):
    result_rows = []
    plus_subscribers = defaultdict(lambda: {
        'is_plus': False,
        'plus.first_subscription': None,
        'plus.last_subscription': None,
        '_updated': None,
    })
    for row in rows:
        if row['uid'] is None:
            continue
        row_uid = int(row['uid'])
        if context.table_index == 0:
            result_rows.append({
                'id': row['id'],
                'uid': row_uid,
            })
        else:
            plus_subscribers[row_uid] = row
    for row in result_rows:
        row_uid = int(row['uid']) if row['uid'] else None
        row.update(plus_subscribers[row_uid])
        del row['uid']
        del row['_updated']
        yield row


@with_context
def _bin_attr_reducer(key, rows, context):
    bin_attr = dict(bin_card=None)
    for row in rows:
        bin_card = row['bin_card']
        if context.table_index == 0:
            bin_attr = row
        else:
            if bin_card == bin_attr['bin_card']:
                yield dict(
                    id=row['id'],
                    bin_attr=bin_attr['bin_attr'],
                )


def _billing_tasks_reducer(key, rows):
    debt_dict = defaultdict(lambda: float())
    for row in rows:
        if row['state'] == 'active':
            debt_dict[row['id']] += row['bill']
    for user_id, debt in debt_dict.items():
        yield {
            'id': user_id,
            'debt': debt,
        }


def _extended_mapper(row):
    register_time = None
    if row['register_time']:
        register_time = (
            datetime.fromtimestamp(row['register_time']).isoformat()
        )
    last_device_id = None
    if row['last_device_id']:
        last_device_id = row['last_device_id']
    yield dict(
        id=row['user_id'],
        first_order=(
            row['first_order_time'] if row['first_order_time'] > 0 else None
        ),
        last_order=(
            row['last_order_time'] if row['last_order_time'] > 0 else None
        ),
        registered_at=register_time,
        device_id=last_device_id,
        app_list=row['apps'],
    )


class UsersExportJob(YtCronJob):
    code = 'users_export'

    def _update_large_table(self):
        LOGGER.info('Merging all tables in one')
        with self._yt.Transaction(timeout=self.transaction_timeout):
            # Migration step
            # Prepare data from new export
            self._yt.run_map(
                _extended_mapper,
                source_table='//home/carsharing/production/data/user/extended',
                destination_table='//home/carsharing/production/data/user/_parts/extended',
            )
            self._yt.run_sort(
                source_table='//home/carsharing/production/data/user/_parts/extended',
                destination_table='//home/carsharing/production/data/user/_parts/extended',
                sort_by=['id'],
            )
            # Step 0
            # Get list of all parts
            table_parts = [
                '//home/carsharing/production/data/user/_parts/extended',
                settings.EXPORT['users_table_part'],
                settings.EXPORT['orders_table_part'],
                settings.EXPORT['bonus_account_table_part'],
                settings.EXPORT['autocode_fine_table_part'],
                settings.EXPORT['billing_tasks_table_part'],
                settings.EXPORT['datasync_part'],
                settings.EXPORT['security_part'],
                settings.EXPORT['license_loss_part'],
                settings.EXPORT['users_bin_attr_part'],
                settings.EXPORT['user_profiles_part'],
#                 settings.EXPORT['users_apps_part'],
                settings.EXPORT['users_crypta_part'],
                settings.EXPORT['users_crypta_profiles_part'],
                settings.EXPORT['current_score_part'],
                settings.EXPORT['plus_subscribers_part'],
                UserTags()._get_path(),
            ]
            # Maybe some tables are broken, so merge only good tables
            # HINT: Broken table --- non-existent table
            table_parts = list(filter(self._yt.exists, table_parts))
            # Step 1
            # Prepare table schema
            schema = [
                {'name': 'id',                          'type': 'string'},
                {'name': 'uid',                         'type': 'int64'},
                {'name': 'uuid',                        'type': 'string'},
                {'name': 'device_id',                   'type': 'string'},
                {'name': 'email',                       'type': 'string'},
                {'name': 'is_yandexoid',                'type': 'boolean'},
                {'name': 'masked_card',                 'type': 'string'},
                {'name': 'bin_card',                    'type': 'string'},
                {'name': 'phone',                       'type': 'string'},
                {'name': 'platform',                    'type': 'string'},
                {'name': 'registered_at',               'type': 'string'},
                {'name': 'registration_chat.action_id', 'type': 'string'},
                {'name': 'registration_chat.status',    'type': 'string'},
                {'name': 'status',                      'type': 'string'},
                {'name': 'tag_list',                    'type': 'any'},
                {'name': 'role_list',                   'type': 'any'},
                {'name': 'region_id',                   'type': 'string'},
                {'name': 'total_cost_fine',             'type': 'double'},
                {'name': 'total_paid_fine',             'type': 'double'},
                {'name': 'total_count_of_fines',        'type': 'int64'},
                {'name': 'debt',                        'type': 'double'},
                # DataSync part
                {'name': 'age',                         'type': 'int64'},
                {'name': 'gender',                      'type': 'string'},
                {'name': 'driver_license',              'type': 'string'},
                {'name': 'issue_date',                  'type': 'int64'},
                {'name': 'valid_to_date',               'type': 'int64'},
                {'name': 'experience_from',             'type': 'int64'},
                {'name': 'experience',                  'type': 'int64'},
                {'name': 'birth_date_dl',               'type': 'int64'},
                {'name': 'first_name_dl',               'type': 'string'},
                {'name': 'last_name_dl',                'type': 'string'},
                {'name': 'patronymic_name_dl',          'type': 'string'},
                {'name': 'passport',                    'type': 'string'},
                {'name': 'birth_date_pass',             'type': 'int64'},
                {'name': 'first_name_pass',             'type': 'string'},
                {'name': 'last_name_pass',              'type': 'string'},
                {'name': 'patronymic_name_pass',        'type': 'string'},
                # Orders table part
                {'name': 'first_order',                 'type': 'int64'},
                {'name': 'last_order',                  'type': 'int64'},
                {'name': 'count_of_order',              'type': 'int64'},
                {'name': 'total_paid',                  'type': 'double'},
                {'name': 'total_discount',              'type': 'double'},
                {'name': 'total_bonus',                 'type': 'double'},
                # Bonus account table part
                {'name': 'bonus_account',               'type': 'double'},
                # DL loss part
                {'name': 'dl_loss',                     'type': 'any'},
                # Security part
                {'name': 'total_cost_of_damages',        'type': 'double'},
                {'name': 'total_paid_of_damages',        'type': 'double'},
                {'name': 'total_to_pay_for_damages',     'type': 'double'},
                # Plus subscribers part
                {'name': 'is_plus',                      'type': 'boolean'},
                {'name': 'plus.first_subscription',      'type': 'int64'},
                {'name': 'plus.last_subscription',       'type': 'int64'},
                # Social profiles part
                {'name': 'social_profiles',              'type': 'any'},
                # Bin attr part
                {'name': 'bin_attr',                     'type': 'any'},
                # Current score
                {'name': 'current_score',                'type': 'double'},
                # Users apps part
                {'name': 'app_list',                     'type': 'any'},
                # Users crypta part
                {'name': 'crypta_id',                    'type': 'uint64'},
                {'name': 'yandexuid',                    'type': 'any'},
                {'name': 'gaid',                         'type': 'string'},
                # Users crypta profiles part
                {'name': 'crypta_profile',               'type': 'any'},
                # Users tags
                {'name': 'user_tags',                    'type': 'any'},
                {'name': 'registration_expiration_date', 'type': 'int64'},
                {'name': 'expiration_date',              'type': 'int64'},
            ]
            large_table = TablePath(
                settings.EXPORT['users_table'],
                schema=schema,
            )
            # Step 2
            # Reduce all tables by id
            self._yt.run_reduce(
                _large_table_merger,
                source_table=table_parts,
                destination_table=large_table,
                reduce_by=['id'],
            )
            # Step 3
            # Sort users by id
            self._yt.run_sort(
                source_table=large_table,
                destination_table=large_table,
                sort_by=['id'],
            )

    def _update_datasync_part(self):
        LOGGER.info('Updating datasync part')
        datasync_export = DataSyncExport(self._yt)
        datasync_export.export(
            settings.EXPORT['users_table_part'],
            settings.EXPORT['datasync_part']
        )

    def _update_users_apps_part(self):
        LOGGER.info('Updating apps part')
        with self._yt.Transaction(timeout=self.transaction_timeout):
            sorted_users_table = (
                self._yt.create_temp_table(settings.EXPORT['tmp_path'])
            )
            self._yt.run_sort(
                source_table=settings.EXPORT['users_table_part'],
                destination_table=sorted_users_table,
                sort_by=['device_id'],
            )
            today_state = (
                datetime.utcnow()
                .replace(hour=0, minute=0, second=0, microsecond=0)
            )
            original_apps_table_list = []
            for i in range(7):
                source_table = settings.EXPORT['apps_table_format'].format(
                    (today_state - timedelta(days=i)).strftime('%Y-%m-%d')
                )
                if self._yt.exists(source_table):
                    original_apps_table_list.append(source_table)
            mapped_apps_table = (
                self._yt.create_temp_table(settings.EXPORT['tmp_path'])
            )
            self._yt.run_map(
                _users_apps_mapper,
                original_apps_table_list,
                mapped_apps_table
            )
            self._yt.run_sort(
                source_table=mapped_apps_table,
                destination_table=mapped_apps_table,
                sort_by=['device_id'],
            )
            self._yt.run_reduce(
                _users_apps_reducer,
                source_table=[
                    sorted_users_table,
                    mapped_apps_table,
                ],
                destination_table=mapped_apps_table,
                reduce_by=['device_id'],
            )
            self._yt.run_sort(
                source_table=mapped_apps_table,
                destination_table=mapped_apps_table,
                sort_by=['id'],
            )
            if not self._yt.exists(settings.EXPORT['users_apps_part']):
                self._yt.create(
                    'table',
                    settings.EXPORT['users_apps_part'],
                    recursive=True,
                )
                self._yt.run_sort(
                    source_table=settings.EXPORT['users_apps_part'],
                    destination_table=settings.EXPORT['users_apps_part'],
                    sort_by=['id'],
                )
            self._yt.run_reduce(
                _users_apps_merger,
                source_table=[
                    settings.EXPORT['users_apps_part'],
                    mapped_apps_table,
                ],
                destination_table=settings.EXPORT['users_apps_part'],
                reduce_by=['id'],
            )
            self._yt.remove(mapped_apps_table)
            self._yt.remove(sorted_users_table)
            self._yt.run_sort(
                source_table=settings.EXPORT['users_apps_part'],
                destination_table=settings.EXPORT['users_apps_part'],
                sort_by=['id'],
            )

    def _update_users_crypta_part(self):
        LOGGER.info('Updating crypta part')
        with self._yt.Transaction(timeout=self.transaction_timeout):
            sorted_users_table = (
                self._yt.create_temp_table(settings.EXPORT['tmp_path'])
            )
            self._yt.run_sort(
                source_table=settings.EXPORT['users_table_part'],
                destination_table=sorted_users_table,
                sort_by=['device_id'],
            )
            mapped_crypta_table = (
                self._yt.create_temp_table(settings.EXPORT['tmp_path'])
            )
            self._yt.run_map(
                _users_crypta_mapper,
                settings.EXPORT['crypta_table'],
                mapped_crypta_table
            )
            self._yt.run_sort(
                source_table=mapped_crypta_table,
                destination_table=mapped_crypta_table,
                sort_by=['device_id'],
            )
            self._yt.run_reduce(
                _users_crypta_reducer,
                source_table=[
                    sorted_users_table,
                    mapped_crypta_table,
                ],
                destination_table=mapped_crypta_table,
                reduce_by=['device_id'],
            )
            self._yt.run_sort(
                source_table=mapped_crypta_table,
                destination_table=mapped_crypta_table,
                sort_by=['id'],
            )
            self._yt.run_reduce(
                _users_crypta_merger,
                source_table=[
                    mapped_crypta_table,
                    settings.EXPORT['crypta_gaid_table'],
                    settings.EXPORT['crypta_yandexuid_table'],
                ],
                destination_table=settings.EXPORT['users_crypta_part'],
                reduce_by=['id'],
            )
            self._yt.remove(mapped_crypta_table)
            self._yt.remove(sorted_users_table)
            self._yt.run_sort(
                source_table=settings.EXPORT['users_crypta_part'],
                destination_table=settings.EXPORT['users_crypta_part'],
                sort_by=['id'],
            )

    def _update_database_part(self):
        LOGGER.info('Updating users_table part')
        users_export = UsersExport(self._yt)
        users_export.export(settings.EXPORT['users_table_part'])
        LOGGER.info('Updating orders_table part')
        orders_export = OrdersExport(self._yt)
        orders_export.export(settings.EXPORT['orders_table_part'])
        LOGGER.info('Updating bonus_account_table part')
        bonus_account_export = BonusAccountExport(self._yt)
        bonus_account_export.export(settings.EXPORT['bonus_account_table_part'])
        LOGGER.info('Updating autocode_fine part')
        autocode_fine_export = AutocodeFineExport(self._yt)
        autocode_fine_export.export(settings.EXPORT['autocode_fine_table_part'])
        LOGGER.info('Updating billing_tasks part')
        billing_tasks_export = BillingTasksExport(self._yt)
        billing_tasks_export.export(settings.EXPORT['billing_tasks_table_part'])

    def _update_crypta_profiles_part(self):
        LOGGER.info('Updating crypta profiles part')
        with self._yt.Transaction(timeout=self.transaction_timeout):
            mapped_crypta_table = TablePath(
                self._yt.create_temp_table(settings.EXPORT['tmp_path']),
                schema=[
                    {'name': 'id',        'type': 'string'},
                    {'name': 'yandexuid', 'type': 'uint64'},
                ],
            )
            self._yt.run_map(
                _users_crypta_profiles_mapper,
                source_table=settings.EXPORT['users_crypta_part'],
                destination_table=mapped_crypta_table,
            )
            self._yt.run_sort(
                source_table=mapped_crypta_table,
                destination_table=mapped_crypta_table,
                sort_by=['yandexuid']
            )
            self._yt.run_reduce(
                _users_crypta_profiles_merger,
                source_table=[
                    mapped_crypta_table,
                    settings.EXPORT['crypta_profiles_table'],
                ],
                destination_table=settings.EXPORT['users_crypta_profiles_part'],
                reduce_by=['yandexuid'],
            )
            self._yt.remove(mapped_crypta_table)
            self._yt.run_sort(
                source_table=settings.EXPORT['users_crypta_profiles_part'],
                destination_table=settings.EXPORT['users_crypta_profiles_part'],
                sort_by=['id'],
            )

    def _update_current_score_part(self):
        LOGGER.info('Updating current score part')
        with self._yt.Transaction(timeout=self.transaction_timeout):
            sorted_current_score_table = (
                self._yt.create_temp_table(settings.EXPORT['tmp_path'])
            )
            self._yt.run_sort(
                source_table=settings.EXPORT['current_score_table'],
                destination_table=sorted_current_score_table,
                sort_by=['id'],
            )
            self._yt.run_reduce(
                _current_score_reducer,
                source_table=[
                    settings.EXPORT['users_table_part'],
                    sorted_current_score_table,
                ],
                destination_table=settings.EXPORT['current_score_part'],
                reduce_by=['id'],
            )
            self._yt.run_sort(
                source_table=settings.EXPORT['current_score_part'],
                destination_table=settings.EXPORT['current_score_part'],
                sort_by=['id'],
            )
            self._yt.remove(sorted_current_score_table)

    def _update_plus_subscribers_part(self):
        LOGGER.info('Updating plus subscribers part')
        plus_table = PlusTable(self._yt)
        plus_table.update()
        with self._yt.Transaction(timeout=self.transaction_timeout):
            sorted_users_table = (
                self._yt.create_temp_table(settings.EXPORT['tmp_path'])
            )
            self._yt.run_sort(
                source_table=settings.EXPORT['users_table_part'],
                destination_table=sorted_users_table,
                sort_by=['uid'],
            )
            self._yt.run_reduce(
                _plus_subscribers_reducer,
                source_table=[
                    sorted_users_table,
                    plus_table.get_table_path(),
                ],
                destination_table=settings.EXPORT['plus_subscribers_part'],
                reduce_by=['uid'],
            )
            self._yt.run_sort(
                source_table=settings.EXPORT['plus_subscribers_part'],
                destination_table=settings.EXPORT['plus_subscribers_part'],
                sort_by=['id'],
            )
            self._yt.remove(sorted_users_table)

    def _update_security_part(self):
        LOGGER.info('Updating security part')
        security_export = SecurityExport(self._yt)
        security_export.export(settings.EXPORT['security_part'])

    def _update_user_profiles_part(self):
        LOGGER.info('Updating user_profiles part')
        user_profiles_export = UserProfilesExport(self._yt)
        user_profiles_export.export(settings.EXPORT['user_profiles_part'])

    def _update_license_loss_part(self):
        LOGGER.info('Updating license_loss part')
        license_loss_table = LicenseLossTable(self._yt)
        license_loss_table.update()

    def _update_bin_attr_part(self):
        LOGGER.info('Updating bin_attr part')
        bin_attr_table = BinAttrTable(self._yt)
        bin_attr_table.update()
        with self._yt.Transaction(timeout=self.transaction_timeout):
            sorted_users_table = (
                self._yt.create_temp_table(settings.EXPORT['tmp_path'])
            )
            self._yt.run_sort(
                source_table=settings.EXPORT['users_table_part'],
                destination_table=sorted_users_table,
                sort_by=['bin_card'],
            )
            self._yt.run_join_reduce(
                _bin_attr_reducer,
                source_table=[
                    TablePath(bin_attr_table.get_table_path(), foreign=True),
                    sorted_users_table,
                ],
                destination_table=settings.EXPORT['users_bin_attr_part'],
                join_by=['bin_card'],
            )
            self._yt.run_sort(
                source_table=settings.EXPORT['users_bin_attr_part'],
                destination_table=settings.EXPORT['users_bin_attr_part'],
                sort_by=['id'],
            )
            self._yt.remove(sorted_users_table)

    def _update_user_tags_part(self):
        LOGGER.info('Updating user_tags part')
        user_tags = UserTags()
        user_tags.save(self._yt)

    def _do_tick(self):
        operations = [
            # Step 1
            # Export all data from database
            self._update_database_part,
            # Step 2
            # Export all data from other sources
            self._update_user_tags_part,
            self._update_datasync_part,
            self._update_security_part,
#            self._update_bin_attr_part,
            self._update_user_profiles_part,
            self._update_license_loss_part,
#             self._update_users_apps_part,
            self._update_users_crypta_part,
            self._update_crypta_profiles_part,
            self._update_current_score_part,
            self._update_plus_subscribers_part,
            # Step 3
            # Merge all tables into big one
            self._update_large_table,
        ]

        # Execute all operations
        for operation in operations:
            try:
                operation()
            except Exception as exc:
                LOGGER.exception(exc)
