# -*- coding: utf-8 -*-

import os
import time
import yt.wrapper
from datetime import datetime, timedelta
from cars import settings
from collections import defaultdict
from .table_export import TableExport

# Yt bugfix
import packaging
import packaging.version
import packaging.specifiers
import packaging.requirements

# Yt should not fail now
if os.environ.get('DJANGO_SETTINGS_MODULE') is not None:
    from django.core.exceptions import ObjectDoesNotExist
    from django.db.models import Q
    from cars.billing.models import Task


def _table_reducer(key, rows):
    result = defaultdict(float)
    for row in rows:
        result[row['id']] += row['debt']
    for user_id, debt in result.items():
        yield {
            'id': user_id,
            'debt': debt,
        }


class BillingTasksExport(TableExport):
    @classmethod
    def _get_all_query_set(cls, finish):
        return (
            Task.objects
            .using(settings.DB_RO_ID)
            .filter(
                Q(task_status='no_cards') | Q(task_status='no_funds')
            )
        )

    def _save_table(self, path):
        self._yt.run_sort(
            source_table=path,
            destination_table=path,
            sort_by=self._get_merge_columns(),
        )
        self._yt.run_reduce(
            _table_reducer,
            source_table=path,
            destination_table=path,
            reduce_by=self._get_merge_columns(),
        )

    def _export_all(self, path, finish):
        super()._export_all(path, finish)
        self._save_table(path)

    def _export_delta(self, path, start, finish):
        super()._export_all(path, finish)
        self._save_table(path)

    @classmethod
    def _get_finish(cls):
        # We always reload full table, so, we don't need to know
        # last updated state
        return None

    def _get_columns(self):
        return {
            'id': ('user_id', self._to_string),
            'debt': ('bill', float)
        }

    @classmethod
    def _get_merge_columns(cls):
        return ['id']
