# -*- coding: utf-8 -*-

import os
import time
import yt.wrapper
from collections import defaultdict
from datetime import datetime, timedelta
from cars import settings
from .table_export import TableExport

# Yt bugfix
import packaging
import packaging.version
import packaging.specifiers
import packaging.requirements

# Yt should not fail now
if os.environ.get('DJANGO_SETTINGS_MODULE') is not None:
    from cars.billing.models import BonusAccount, BonusAccountOperation
    from django.core.exceptions import ObjectDoesNotExist


def _accounts_merger(key, rows):
    for row in rows:
        yield row


def _accounts_reducer(key, rows):
    orders = {}
    for row in rows:
        row_id = row['id']
        if row_id in orders:
            orders[row_id]['bonus_account'] += row['bonus_account']
        else:
            orders[row_id] = row
    for order in orders.values():
        yield order


class BonusAccountExport(TableExport):
    @classmethod
    def _get_all_query_set(cls, finish):
        return (
            BonusAccountOperation.objects
            .using(settings.DB_RO_ID)
            .filter(
                created_at__lt=finish
            )
        )

    @classmethod
    def _get_delta_query_set(cls, start, finish):
        return (
            BonusAccountOperation.objects
            .using(settings.DB_RO_ID)
            .filter(
                created_at__gte=start,
                created_at__lt=finish
            )
        )

    @classmethod
    def _get_finish(cls):
        return (datetime.now() - timedelta(minutes=5)).isoformat()

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._accounts = defaultdict(lambda: None)

    def _save_table(self, path):
        self._yt.run_sort(
            source_table=path,
            destination_table=path,
            sort_by=['id'],
        )
        self._yt.run_reduce(
            _accounts_reducer,
            source_table=path,
            destination_table=path,
            reduce_by=['id'],
        )

    def _export_all(self, path, finish):
        self._prefetch_data(None)
        super()._export_all(path, finish)
        self._save_table(path)

    def _export_delta(self, path, start, finish):
        self._prefetch_data(self._get_account_ids(start, finish))
        super()._export_delta(path, start, finish)
        self._save_table(path)

    def _get_account_ids(self, start, finish):
        return (
            self._get_delta_query_set(start, finish)
            .order_by('bonus_account_id')
            .distinct('bonus_account_id')
            .values_list('bonus_account_id', flat=True)
        )

    def _prefetch_data(self, account_ids):
        accounts_qs = (
            BonusAccount.objects
            .using(settings.DB_RO_ID)
        )
        if account_ids:
            accounts_qs = accounts_qs.filter(id__in=account_ids)
        for account in accounts_qs.iterator():
            self._accounts[account.id] = str(account.user_id)

    def _get_columns(self):
        return {
            'id':            ('bonus_account_id', self._get_user_id),
            'bonus_account': ('amount',           float),
        }

    @classmethod
    def _get_merge_columns(cls):
        return ['id']

    @classmethod
    def _get_merger(cls):
        return _accounts_merger

    def _get_user_id(self, account_id):
        return self._accounts[account_id]
