# -*- coding: utf-8 -*-

import os
import logging
import yt.wrapper
from collections import defaultdict
from cars import settings
from yt.wrapper import TablePath
from multiprocessing.dummy import Pool
from ..helpers import hash_passport_parts, hash_user_name

# Yt bugfix
import packaging
import packaging.version
import packaging.specifiers
import packaging.requirements

# Yt should not fail now
if os.environ.get('DJANGO_SETTINGS_MODULE') is not None:
    from cars.users.core.datasync import DataSyncDocumentsClient


LOGGER = logging.getLogger(__name__)


class DataSyncExport(object):
    def __init__(self, yt):
        self._yt = yt

    @staticmethod
    @yt.wrapper.with_context
    def _table_diff_reducer(key, rows, context):
        curr = defaultdict(lambda: {'_passport_key': None, '_license_key': None})
        prev = defaultdict(lambda: {'_passport_key': None, '_license_key': None})
        for row in rows:
            row_id = row['id']
            if context.table_index == 0:
                curr[row_id] = {
                    'id': row_id,
                    'uid': row['uid'],
                    '_passport_key': row['_passport_key'],
                    '_license_key': row['_license_key'],
                }
            else:
                prev[row_id] = row
        for key, row in curr.items():
            if row['_passport_key'] != prev[key]['_passport_key']:
                yield row
            elif row['_license_key'] != prev[key]['_license_key']:
                yield row

    @staticmethod
    @yt.wrapper.with_context
    def _delta_merger(key, rows, context):
        result = {}
        for row in rows:
            row_id = row['id']
            if context.table_index == 0:
                if row_id not in result:
                    result[row_id] = row
            else:
                result[row_id] = row
        for row in result.values():
            yield row

    def export(self, users_table, path):
        with self._yt.TempTable(settings.EXPORT['tmp_path']) as sorted_users_table:
            self._yt.run_sort(
                source_table=users_table,
                destination_table=sorted_users_table,
                sort_by=['id'],
            )
            if not self._yt.exists(path):
                self._yt.create("table", path, ignore_existing=True)
                self._yt.run_sort(
                    source_table=path,
                    destination_table=path,
                    sort_by=['id'],
                )
            with self._yt.TempTable(settings.EXPORT['tmp_path']) as diff_table:
                # Compare current state with latest datasync table
                self._yt.run_reduce(
                    self._table_diff_reducer,
                    source_table=[
                        sorted_users_table,
                        path,
                    ],
                    destination_table=diff_table,
                    reduce_by=['id'],
                )
                with self._yt.TempTable(settings.EXPORT['tmp_path']) as delta_table:
                    # Load new data to delta table
                    delta_data = list(self._yt.read_table(diff_table))
                    diff_data = self._load_data(delta_data)
                    self._yt.write_table(
                        TablePath(delta_table, append=True),
                        iter(diff_data)
                    )
                    self._yt.run_sort(
                        source_table=delta_table,
                        destination_table=delta_table,
                        sort_by=['id'],
                    )
                    with self._yt.Transaction():
                        # Merge delta with datasync table
                        self._yt.run_reduce(
                            self._delta_merger,
                            source_table=[
                                path,
                                delta_table,
                            ],
                            destination_table=path,
                            reduce_by=['id'],
                        )
                        # DataSync table should be sorted by ID
                        self._yt.run_sort(
                            source_table=path,
                            destination_table=path,
                            sort_by=['id'],
                        )

    def _load_data(self, rows):
        self._datasync_client = DataSyncDocumentsClient.from_settings()
        pool = Pool(8)
        result = pool.map(self._load_row_data, rows)
        pool.close()
        pool.join()
        return result

    def _load_row_data(self, row):
        row.update({
            'gender': None,
            'driver_license': None,
            'experience_from': None,
            'issue_date': None,
            'valid_to_date': None,
            'birth_date_dl': None,
            'first_name_dl': None,
            'last_name_dl': None,
            'patronymic_name_dl': None,
            'passport': None,
            'birth_date_pass': None,
            'first_name_pass': None,
            'last_name_pass': None,
            'patronymic_name_pass': None,
        })
        try:
            if row['_passport_key']:
                data = self._datasync_client.get_passport_unverified(
                    row['uid'],
                    row['_passport_key']
                )
                row['_passport_key'] = '?' + row['_passport_key']
                if data:
                    row.update({
                        'gender': data.get('gender'),
                        'passport': hash_passport_parts(
                            data.get('doc_value'),
                            data.get('subdivision_code')
                        ),
                        'birth_date_pass': data.get('birth_date'),
                        'first_name_pass': hash_user_name(data.get('first_name')),
                        'last_name_pass': hash_user_name(data.get('last_name')),
                        'patronymic_name_pass': hash_user_name(data.get('middle_name')),
                        'registration_expiration_date': data.get('registration_expiration_date'),
                        'expiration_date': data.get('expiration_date'),
                    })
                    row['_passport_key'] = str(row['_passport_key'][1:])
        except Exception as exc:
            LOGGER.exception(exc)
        try:
            if row['_license_key']:
                data = self._datasync_client.get_license_unverified(
                    row['uid'],
                    row['_license_key']
                )
                row['_license_key'] = '?' + row['_license_key']
                if data:
                    row.update({
                        'driver_license':
                            data.get('number_front') or data.get('number_back'),
                        'experience_from': data.get('experience_from'),
                        'issue_date': data.get('issue_date'),
                        'valid_to_date': data.get('categories_b_valid_to_date'),
                        'birth_date_dl': data.get('birth_date'),
                        'first_name_dl': hash_user_name(data.get('first_name')),
                        'last_name_dl': hash_user_name(data.get('last_name')),
                        'patronymic_name_dl': hash_user_name(data.get('middle_name')),
                    })
                    row['_license_key'] = str(row['_license_key'][1:])
        except Exception as exc:
            LOGGER.exception(exc)
        return row
