# -*- coding: utf-8 -*-

import os
import time
import yt.wrapper
from datetime import datetime, timedelta
from cars import settings
from .table_export import TableExport

# Yt bugfix
import packaging
import packaging.version
import packaging.specifiers
import packaging.requirements

# Yt should not fail now
if os.environ.get('DJANGO_SETTINGS_MODULE') is not None:
    from cars.users.models import User
    from django.core.exceptions import ObjectDoesNotExist
    from cars.orders.models import CompiledRide


def _orders_merger(key, rows):
    for row in rows:
        yield row


def _orders_reducer(key, rows):
    orders = {}
    for row in rows:
        row_id = row['id']
        if row_id in orders:
            orders[row_id]['first_order'] = min(
                orders[row_id]['first_order'],
                row['first_order']
            )
            orders[row_id]['last_order'] = max(
                orders[row_id]['last_order'],
                row['last_order']
            )
            orders[row_id]['total_paid'] += row['total_paid']
            orders[row_id]['total_bonus'] += row['total_bonus']
            orders[row_id]['total_discount'] += row['total_discount']
            orders[row_id]['count_of_order'] += row['count_of_order']
        else:
            orders[row_id] = row
    for order in orders.values():
        yield order


class OrdersExport(TableExport):
    @classmethod
    def _get_all_query_set(cls, finish):
        return (
            CompiledRide.objects
            .using(settings.DB_RO_ID)
            .filter(history_timestamp__lt=finish)
        )

    @classmethod
    def _get_delta_query_set(cls, start, finish):
        return (
            CompiledRide.objects
            .using(settings.DB_RO_ID)
            .filter(
                history_timestamp__gte=start,
                history_timestamp__lt=finish
            )
        )

    def _save_table(self, path):
        self._yt.run_sort(
            source_table=path,
            destination_table=path,
            sort_by=['id'],
        )
        self._yt.run_reduce(
            _orders_reducer,
            source_table=path,
            destination_table=path,
            reduce_by=['id'],
        )

    def _export_all(self, path, finish):
        super()._export_all(path, finish)
        self._save_table(path)

    def _export_delta(self, path, start, finish):
        super()._export_delta(path, start, finish)
        self._save_table(path)

    @classmethod
    def _get_finish(cls):
        return int(time.time()) - 60 * 5

    def _get_columns(self):
        return {
            'id': ('history_user_id', self._to_string),
            'first_order': ('history_timestamp', int),
            'last_order': ('history_timestamp', int),
            'count_of_order': ('history_user_id', self._get_count_one),
        }

    def _get_row_extra(self, obj):
        total_paid = float()
        total_bonus = float()
        total_discount = float()
        if obj.meta and 'bill' in obj.meta:
            bill = obj.meta['bill']
            records = bill
            if isinstance(bill, dict):
                records = bill.get('records', [])
            for record in records:
                record_type = record.get('type')
                if record_type == 'total':
                    total_paid = float(record.get('cost'))
                elif record_type == 'discount':
                    record_cost = abs(float(record.get('cost', 0)))
                    total_discount += record_cost
                elif record_type == 'billing_bonus':
                    record_cost = abs(float(record.get('cost', 0)))
                    total_bonus += record_cost
        return {
            'total_paid': total_paid,
            'total_bonus': total_bonus,
            'total_discount': total_discount,
        }

    @classmethod
    def _get_merge_columns(cls):
        return ['id']

    @classmethod
    def _get_merger(cls):
        return _orders_merger

    @staticmethod
    def _get_count_one(value):
        return 1
