# -*- coding: utf-8 -*-

import os
import json
import logging
import requests
import yt.wrapper
from re import compile
from collections import defaultdict
from cars import settings
from yt.wrapper import TablePath

# Yt bugfix
import packaging
import packaging.version
import packaging.specifiers
import packaging.requirements


LOGGER = logging.getLogger(__name__)


def _security_reducer(key, rows):
    security = {}
    for row in rows:
        row_id = row['id']
        if row_id in security:
            security[row_id]['total_cost_of_damages'] += row['total_cost_of_damages']
            security[row_id]['total_to_pay_for_damages'] += row['total_to_pay_for_damages']
            security[row_id]['total_paid_of_damages'] += row['total_paid_of_damages']
        else:
            security[row_id] = row
    for row in security.values():
        yield row


class SecurityExport(object):
    TRANSACTION_TIMEOUT = 1000 * 60 * 60 * 12
    #                     ^ms    ^s   ^m   ^h

    USER_ID_EXPR = compile(r'\/clients\/([a-fA-F0-9\-]+)\/')

    def __init__(self, yt):
        self._yt = yt

    def export(self, path):
        with self._yt.Transaction(timeout=self.TRANSACTION_TIMEOUT):
            self._yt.write_table(path, iter(self._get_data()))
            self._yt.run_sort(path, sort_by=['id'])
            self._yt.run_reduce(
                _security_reducer,
                source_table=path,
                destination_table=path,
                reduce_by=['id'],
            )
            self._yt.run_sort(path, sort_by=['id'])

    def _get_keys(self):
        data = self._send_http_request(
            settings.EXPORT['security_issues_list_url']
        )
        return (item['key'] for item in data)

    def _get_data(self):
        result = []
        for key in self._get_keys():
            data = self._send_http_request(
                settings.EXPORT['security_issue_url_format'].format(key)
            )
            match = self.USER_ID_EXPR.search(data.get('description', ''))
            if not match:
                continue
            user_id = match.group(1)
            cost_of_damage = data.get('amountOfDamageFromCarService')
            if cost_of_damage is None:
                cost_of_damage = data.get('amountPaidByInsurance', 0.0)
            result.append({
                'id': user_id,
                'total_cost_of_damages': cost_of_damage,
                'total_to_pay_for_damages': data.get('amountToPayByClient', 0.0),
                'total_paid_of_damages': data.get('amountPaidByClient', 0.0),
            })
        return result

    def _send_http_request(self, url):
        headers = {
            'Authorization': 'OAuth {}'.format(settings.STARTREK['token']),
            'Content-Type': 'application/json'
        }
        try:
            response = requests.get(
                settings.STARTREK['api_url'] + url,
                headers=headers,
                verify=False,
            )
            if response.status_code == 200:
                return json.loads(response.content.decode('utf-8'))
        except Exception as exc:
            pass
        return {}
