# -*- coding: utf-8 -*-

from cars import settings
from yt.wrapper import TablePath

# Yt bugfix
import packaging
import packaging.version
import packaging.specifiers
import packaging.requirements


class TableExport(object):
    transaction_timeout = 1000 * 60 * 60 * 12
    #                     ^ms    ^s   ^m   ^h

    def __init__(self, yt):
        self._yt = yt

    def export(self, path):
        with self._yt.Transaction(timeout=self.transaction_timeout):
            finish = self._get_finish()
            if self._yt.has_attribute(path, 'finish'):
                start = self._yt.get_attribute(path, 'finish')
                self._export_delta(path, start, finish)
                self._yt.set_attribute(path, 'start', start)
            else:
                self._export_all(path, finish)
            self._yt.run_sort(
                source_table=path,
                destination_table=path,
                sort_by=self._get_merge_columns(),
            )
            self._yt.set_attribute(path, 'finish', finish)

    def _export_all(self, path, finish):
        query_set = self._get_all_query_set(finish)
        self._yt.write_table(path, self._extract(query_set))

    def _export_delta(self, path, start, finish):
        query_set = self._get_delta_query_set(start, finish)
        delta_table = self._yt.create_temp_table(settings.EXPORT['tmp_path'])
        self._yt.write_table(delta_table, self._extract(query_set))
        if self._get_merge_columns():
            self._yt.run_sort(
                source_table=delta_table,
                destination_table=delta_table,
                sort_by=self._get_merge_columns(),
            )
        table_path = TablePath(
            path,
            schema=self._get_schema(),
        )
        self._yt.run_reduce(
            self._get_merger(),
            source_table=[path, delta_table],
            destination_table=table_path,
            reduce_by=self._get_merge_columns(),
        )
        self._yt.remove(delta_table)

    def _extract(self, query_set):
        columns = self._get_columns().items()
        for obj in self._get_query_set_iterator(query_set):
            row = {}
            for dst, (src, cast) in columns:
                row[dst] = cast(getattr(obj, src))
            for dst, value in self._get_row_extra(obj).items():
                row[dst] = value
            yield row

    @classmethod
    def _get_finish(cls):
        raise NotImplementedError()

    @classmethod
    def _get_all_query_set(cls, finish):
        raise NotImplementedError()

    @classmethod
    def _get_delta_query_set(cls, start, finish):
        raise NotImplementedError()

    @classmethod
    def _get_query_set_iterator(cls, query_set):
        return query_set.iterator()

    def _get_columns(self):
        return {}

    def _get_row_extra(self, obj):
        return {}

    @classmethod
    def _get_schema(cls):
        return None

    @classmethod
    def _get_merge_columns(cls):
        return []

    @classmethod
    def _get_merger(cls):
        raise NotImplementedError()

    @staticmethod
    def _to_int(value):
        return int(value) if value is not None else None

    @staticmethod
    def _to_bool(value):
        return bool(value) if value is not None else None

    @staticmethod
    def _to_string(value):
        return str(value) if value else None

    @staticmethod
    def _to_isoformat(value):
        return value.isoformat() if value else None

    @staticmethod
    def _to_list(value):
        return value or []
