# -*- coding: utf-8 -*-

import os
import json
import logging
import requests
import yt.wrapper
from re import compile
from collections import defaultdict
from cars import settings
from yt.wrapper import TablePath
from cars.core.tvm2 import TVM2ServiceTicket

# Yt bugfix
import packaging
import packaging.version
import packaging.specifiers
import packaging.requirements


LOGGER = logging.getLogger(__name__)


class UserProfilesExport(object):
    TRANSACTION_TIMEOUT = 1000 * 60 * 60 * 12
    #                     ^ms    ^s   ^m   ^h

    BATCH_SIZE = 200

    service_ticket = TVM2ServiceTicket(
        host=settings.SOCIAL['tvm2']['host'],
        source=settings.SOCIAL['tvm2']['source'],
        destination=settings.SOCIAL['tvm2']['destination'],
        secret=settings.SOCIAL['tvm2']['secret'],
    )

    def __init__(self, yt):
        self._yt = yt

    def export(self, path):
        with self._yt.Transaction(timeout=self.TRANSACTION_TIMEOUT):
            self._yt.write_table(path, iter(self._get_data()))
            self._yt.run_sort(path, sort_by=['id'])

    def _get_data(self):
        keys = self._get_keys()
        profiles = {}
        batch = []
        for key in keys:
            batch.append(key)
            if len(batch) == self.BATCH_SIZE:
                profiles.update(self._get_batch_data(batch))
                batch = []
        if batch:
            profiles.update(self._get_batch_data(batch))
        for key in keys:
            if key['uid'] in profiles:
                yield {
                    'id': key['id'],
                    'social_profiles': profiles[key['uid']],
                }

    def _get_batch_data(self, keys):
        uids = set()
        for key in keys:
            uids.add(key['uid'])
        data = self._send_http_request(
            settings.EXPORT['social_profiles_url_format'].format(','.join(map(str, uids)))
        )
        profiles = defaultdict(list)
        for profile in data.get('profiles', []):
            uid = profile.get('uid', None)
            if uid is None:
                continue
            for address in profile.get('addresses', []):
                profiles[int(uid)].append(address)
        return profiles

    def _get_keys(self):
        result = []
        for row in self._yt.read_table(settings.EXPORT['users_table_part']):
            result.append({
                'id': row['id'],
                'uid': row['uid'],
            })
        return result

    def _send_http_request(self, url):
        headers = {
            'X-Ya-Service-Ticket': self.service_ticket.content,
            'Content-Type': 'application/json',
        }
        params = {
            'consumer': settings.SOCIAL['consumer'],
        }
        try:
            response = requests.get(
                settings.SOCIAL['url'] + url,
                headers=headers,
                params=params,
                verify=False,
            )
            if response.status_code == 200:
                return json.loads(response.content.decode('utf-8'))
        except Exception as exc:
            pass
        return {}
