# -*- coding: utf-8 -*-

import os
import time
import yt.wrapper
from collections import defaultdict
from datetime import datetime, timedelta
from cars import settings
from .table_export import TableExport

# Yt bugfix
import packaging
import packaging.version
import packaging.specifiers
import packaging.requirements

# Yt should not fail now
if os.environ.get('DJANGO_SETTINGS_MODULE') is not None:
    from cars.users.models import User, UserRole, UserRolesHistory, UserCreditCard, AppInstall
    from cars.registration.models import RegistrationChatActionResult
    from django.core.exceptions import ObjectDoesNotExist
    from django.db.models import Q


@yt.wrapper.with_context
def _users_merger(key, rows, context):
    users = {}
    for row in rows:
        row_id = row['id']
        if context.table_index == 1:
            users[row_id] = row
        else:
            if row_id not in users:
                users[row_id] = row
    for user in users.values():
        yield user


class UsersExport(TableExport):
    @classmethod
    def _get_all_query_set(cls, finish):
        return (
            User.objects
            .using(settings.DB_RO_ID)
            .filter(
                updated_at__lt=finish
            )
        )

    @classmethod
    def _get_delta_query_set(cls, start, finish):
        roles_user_ids = (
            UserRolesHistory.objects
            .using(settings.DB_RO_ID)
            .filter(
                history_timestamp__gte=start,
                history_timestamp__lt=finish
            )
            .order_by('user_id')
            .distinct('user_id')
            .values_list('user_id', flat=True)
        )
        return (
            User.objects
            .using(settings.DB_RO_ID)
            .filter(
                Q(id__in=roles_user_ids) |
                (Q(updated_at__gte=start) & Q(updated_at__lt=finish))
            )
        )

    @classmethod
    def _get_finish(cls):
        return int(time.time()) - 60 * 5

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        # self._apps = defaultdict(lambda: {
        #     'device_id': None,
        #     'platform': None,
        #     'uuid': None,
        # })
        # self._chats = defaultdict(lambda: {
        #     'registration_chat.status': 'not_started',
        #     'registration_chat.action_id': None,
        # })
        self._cards = defaultdict(lambda: {
            'masked_card': None,
            'bin_card': None,
        })
        self._roles = defaultdict(list)

    def _export_all(self, path, finish):
        self._prefetch_data(None)
        super()._export_all(path, finish)

    def _export_delta(self, path, start, finish):
        self._export_all(path, finish)
#         self._prefetch_data(self._get_user_ids(start, finish))
#         super()._export_delta(path, start, finish)

    def _get_user_ids(self, start, finish):
        return (
            self._get_delta_query_set(start, finish)
            .values_list('id', flat=True)
        )

    def _prefetch_data(self, user_ids):
        # apps_qs = (
        #     AppInstall.objects
        #     .using(settings.DB_RO_ID)
        #     .order_by('user_id', '-created_at')
        #     .distinct('user_id')
        # )
        # chats_qs = (
        #     RegistrationChatActionResult.objects
        #     .using(settings.DB_RO_ID)
        #     .order_by('user_id', '-submitted_at')
        #     .distinct('user_id')
        # )
        cards_qs = (
            UserCreditCard.objects
            .using(settings.DB_RO_ID)
        )
        roles_qs = (
            UserRole.objects
            .using(settings.DB_RO_ID)
        )
        if user_ids:
            apps_qs = apps_qs.filter(user_id__in=user_ids)
            # chats_qs = chats_qs.filter(user_id__in=user_ids)
            cards_qs = cards_qs.filter(user_id__in=user_ids)
            roles_qs = roles_qs.filter(user_id__in=user_ids)
        # for app in apps_qs.iterator():
        #     self._apps[str(app.user_id)] = {
        #         'device_id': str(app.device_id),
        #         'platform': app.platform,
        #         'uuid': str(app.uuid),
        #     }
        # for chat in chats_qs.iterator():
        #     chat_status = 'completed'
        #     if chat.completed_at is None:
        #         chat_status = 'in_progress'
        #     self._chats[str(chat.user_id)] = {
        #         'registration_chat.status': chat_status,
        #         'registration_chat.action_id': chat.chat_action_id,
        #     }
        for card in cards_qs.iterator():
            self._cards[str(card.user_id)] = {
                'masked_card': card.get_masked_pan(),
                'bin_card': card.pan_prefix,
            }
        for role in roles_qs.iterator():
            self._roles[str(role.user_id)].append(role.role_id)

    def _get_columns(self):
        return {
            'id':            ('id',            self._to_string),
            'uid':           ('uid',           self._to_int),
            'status':        ('status',        self._to_string),
            'registered_at': ('registered_at', self._to_isoformat),
            # 'is_plus':       ('is_plus_user',  bool),
            'is_yandexoid':  ('is_yandexoid',  self._to_bool),
            # 'plus.activated_at': ('introscreens', self._get_plus_activated_at),
            'email':         ('email',         self._to_string),
            'phone':         ('phone',         self._to_string),
            'tag_list':      ('tags',          self._to_list),
            'role_list':     ('id',            self._get_roles),
            'region_id':     ('registration_geo', self._to_string),
            # Used by DataSync export
            '_passport_key': ('passport_ds_revision', self._to_string),
            '_license_key':  ('driving_license_ds_revision', self._to_string),
        }

    def _get_row_extra(self, obj):
        result = {}
        # result.update(self._apps[str(obj.id)])
        # result.update(self._chats[str(obj.id)])
        result.update(self._cards[str(obj.id)])
        return result

    @classmethod
    def _get_merge_columns(cls):
        return ['id']

    @classmethod
    def _get_merger(cls):
        return _users_merger

    def _get_roles(self, user_id):
        return self._roles[str(user_id)]
