# -*- coding: utf-8 -*-

from datetime import datetime, timedelta
from multiprocessing.dummy import Pool
from requests import post
from json import loads
from yt.wrapper import with_context
from cars import settings
from .table import YtTable

class BinAttrTable(YtTable):
    SOURCE_URL = 'https://psm7.com/bin/worker.php'

    TRY_COUNT = 5

    def get_name(self):
        return settings.EXPORT['bin_attr_part']

    def get_schema(self):
        return [
            {'name': 'bin_card', 'type': 'string'},
            {'name': 'bin_attr', 'type': 'any'}
        ]

    def get_index(self):
        return ['bin_card']

    def update(self):
        with self._yt.Transaction():
            today_state = (
                datetime.utcnow()
                .replace(hour=0, minute=0, second=0, microsecond=0)
            )
            last_state = self.get_last_state()
            if last_state:
                return
            if not self._yt.exists(self.get_table_path()):
                self._yt.create(
                    'table',
                    self.get_table_path(),
                    recursive=True,
                    ignore_existing=True,
                )
                self._yt.run_sort(
                    source_table=self.get_table_path(),
                    destination_table=self.get_table_path(),
                    sort_by=self.get_index(),
                )
            bin_attr = list(self._get_bin_attr())
            delta_table = self._yt.create_temp_table(
                settings.EXPORT['temp_path']
            )
            self._yt.write_table(delta_table, bin_attr)
            self._yt.run_sort(
                source_table=delta_table,
                destination_table=delta_table,
                sort_by=self.get_index(),
            )
            self._yt.run_reduce(
                self._table_merger,
                source_table=[
                    self.get_table_path(),
                    delta_table,
                ],
                destination_table=self.get_table_path(),
                reduce_by=['bin_card'],
            )
            self._yt.remove(delta_table)
            self._yt.run_sort(
                source_table=self.get_table_path(),
                destination_table=self.get_table_path(),
                sort_by=self.get_index(),
            )
            self.set_last_state(today_state)

    def get_last_state(self):
        last_state = super().get_last_state()
        if last_state:
            return datetime.strptime(last_state, '%Y-%m-%d')
        return None

    def set_last_state(self, state):
        if state:
            state = state.strftime('%Y-%m-%d')
        super().set_last_state(state)

    @staticmethod
    def _table_reducer(key, rows):
        result = set()
        for row in rows:
            if row['bin_card']:
                result.add(row['bin_card'])
        for row in result:
            yield {'bin_card': row}

    @staticmethod
    @with_context
    def _table_merger(key, rows, context):
        result = {}
        for row in rows:
            if context.table_index != 0 or row['bin_card'] not in result:
                result[row['bin_card']] = row
        for row in result.values():
            yield row

    def _get_bin_card(self):
        bin_list_table = self._yt.create_temp_table(
            settings.EXPORT['temp_path']
        )
        self._yt.run_sort(
            source_table=settings.EXPORT['users_table_part'],
            destination_table=bin_list_table,
            sort_by=['bin_card'],
        )
        self._yt.run_reduce(
            self._table_reducer,
            source_table=bin_list_table,
            destination_table=bin_list_table,
            reduce_by=['bin_card'],
        )
        bin_list_rows = list(
            self._yt.read_table(bin_list_table)
        )
        self._yt.remove(bin_list_table)
        for row in bin_list_rows:
            yield row['bin_card']

    def _get_bin_attr(self):
        bin_cards = list(self._get_bin_card())
        pool = Pool(16)
        bin_attr = pool.map(self._send_http_request, bin_cards)
        pool.close()
        pool.join()
        return filter(None.__ne__, bin_attr)

    def _send_http_request(self, bin_card):
        data = {
            'bin-input': bin_card,
        }
        for try_number in range(self.TRY_COUNT):
            try:
                response = post(self.SOURCE_URL, data)
                if response.status_code == 200:
                    content = loads(response.content.decode('utf-8'))
                    if 'result' in content and content['result'] == 'error':
                        break
                    return {
                        'bin_card': bin_card,
                        'bin_attr': content,
                    }
            except Exception as exc:
                pass
        return None
