# -*- coding: utf-8 -*-

import logging
from datetime import datetime, timedelta
from yt.wrapper import TablePath
from .table import YtTable


LOGGER = logging.getLogger(__name__)


class DailyTable(YtTable):
    def update(self):
        self._day_begin = self.get_last_state()
        if not self._day_begin:
            self._day_begin = self.get_first_day()
        while self._day_begin < datetime.utcnow():
            self._day_end = self._day_begin + timedelta(days=1)
            self._update_day()
            self.set_last_state(self._day_begin)
            self._day_begin = self._day_end

    def _update_day(self):
        raise NotImplementedError()

    def get_dirname(self):
        raise NotImplementedError()

    def get_first_day(self):
        return datetime.utcnow().replace(
            hour=0,
            minute=0,
            second=0,
            microsecond=0,
        )

    def get_name(self):
        return '{}/{}'.format(
            self.get_dirname().rstrip('/'),
            self._day_begin.strftime('%Y-%m-%d')
        )

    def get_table_path(self):
        return TablePath(
            name=self.get_name(),
            schema=self.get_schema(),
        )

    def get_last_state(self):
        state = self._yt.get_attribute(
            path=self.get_dirname(),
            attribute='_last_state',
            default=None,
        )
        if state:
            return datetime.strptime(state, '%Y-%m-%d')
        return self.get_first_day()

    def set_last_state(self, state):
        if state:
            state = state.strftime('%Y-%m-%d')
        self._yt.set_attribute(
            path=self.get_dirname(),
            attribute='_last_state',
            value=state,
        )
