# -*- coding: utf-8 -*-

from datetime import datetime, timedelta
from cars import settings
from .table import YtTable
from yt.wrapper import TablePath, with_context


class LicenseLossTable(YtTable):
    SOURCE_TABLE = '//home/carsharing/production/frod/decis/log/decis'
    START_TABLE = '2018-01-30'

    def get_name(self):
        return settings.EXPORT['license_loss_part']

    def get_schema(self):
        return [
            {'name': 'id',      'type': 'string'},
            {'name': 'dl_loss', 'type': 'any'},
        ]

    def get_index(self):
        return ['id']

    @staticmethod
    def _source_table_mapper(row):
        start = int(
            datetime
            .strptime(row['decis_data'], '%d.%m.%Y')
            .timestamp()
        )
        if row['decis_end']:
            finish = int(
                datetime
                .strptime(row['decis_end'], '%d.%m.%Y')
                .timestamp()
            )
        else:
            finish = None
        yield {
            'id': row['id'],
            'dl_loss': [(start, finish)],
        }

    @staticmethod
    def _table_reducer(key, rows):
        result = dict()
        for row in rows:
            if row['id'] in result:
                result[row['id']].extend(row['dl_loss'])
            else:
                result[row['id']] = row['dl_loss']
        for user_id, dl_loss in result.items():
            dl_loss = list(set(tuple(p) for p in dl_loss))
            dl_loss.sort(key=lambda p: p[0])
            yield {
                'id': user_id,
                'dl_loss': dl_loss
            }       

    def update(self):
        with self._yt.Transaction():
            if not self._yt.exists(self.get_table_path()):
                self._yt.create(
                    'table',
                    self.get_table_path(),
                    recursive=True,
                )
                self._yt.run_sort(
                    source_table=self.get_table_path(),
                    destination_table=self.get_table_path(),
                    sort_by=self.get_index(),
                )
            last_state = self.get_last_state()
            today_state = (
                datetime.utcnow()
                .replace(hour=0, minute=0, second=0, microsecond=0)
            )
            delta_table = self._yt.create_temp_table(
                settings.EXPORT['temp_path']
            )
            source_table_list = []
            while last_state < today_state:
                source_table = '{}/{}'.format(
                    self.SOURCE_TABLE,
                    last_state.strftime('%Y-%m-%d')
                )
                if self._yt.exists(source_table):
                    source_table_list.append(source_table)
                last_state += timedelta(days=1)
            self._yt.run_map(
                self._source_table_mapper,
                source_table_list,
                delta_table
            )
            self._yt.run_sort(
                source_table=delta_table,
                destination_table=delta_table,
                sort_by=self.get_index(),
            )
            self._yt.run_reduce(
                self._table_reducer,
                source_table=[
                    self.get_table_path(),
                    delta_table,
                ],
                destination_table=self.get_table_path(),
                reduce_by=self.get_index(),
            )
            self._yt.remove(delta_table)
            self.set_last_state(last_state - timedelta(days=2))
            self._yt.run_sort(
                source_table=self.get_table_path(),
                destination_table=self.get_table_path(),
                sort_by=self.get_index(),
            )

    def get_last_state(self):
        return datetime.strptime(
            super().get_last_state(self.START_TABLE),
            '%Y-%m-%d'
        )

    def set_last_state(self, state):
        if state:
            state = state.strftime('%Y-%m-%d')
        super().set_last_state(state)
