# -*- coding: utf-8 -*-

from yt.wrapper import TablePath
from cars import settings
from cars.export.yt import Table
from cars.export.tables.cars import CarTag


class CarTagsQueue(Table):
    def __init__(self, **kwargs):
        super().__init__()
        self.__recalc = kwargs.get('recalc', False)

    def _value_to_row(self, value):
        return {
            'event_id': value.event_id,
            'timestamp': value.timestamp,
            'car_id': value.car_id,
            'user_id': value.user_id,
            'tag': value.tag,
            'action': value.action,
            'data': value.data,
            'snapshot': value.snapshot,
        }

    def _row_to_value(self, row):
        return CarTag(
            event_id=row['event_id'],
            timestamp=row['timestamp'],
            car_id=row['car_id'],
            user_id=row['user_id'],
            tag=row['tag'],
            action=row['action'],
            data=row['data'],
            snapshot=row['snapshot'],
        )

    def _get_path(self):
        path = '{}/{}'.format(
            settings.EXPORT['home_directory'],
            'data/orders/_helpers/car_tags_queue'
        )
        if self.__recalc:
            path += '_recalc'
        return path

    def _get_schema(self):
        return [
            {'name': 'event_id',  'type': 'int64',  'required': True},
            {'name': 'timestamp', 'type': 'int64',  'required': True},
            {'name': 'car_id',    'type': 'string', 'required': True},
            {'name': 'user_id',   'type': 'string', 'required': True},
            {'name': 'tag',       'type': 'string', 'required': True},
            {'name': 'action',    'type': 'string', 'required': True},
            {'name': 'data',      'type': 'any'},
            {'name': 'snapshot',  'type': 'any'},
        ]

    def push_table(self, table, client):
        def mapper(row):
            yield self._value_to_row(table._row_to_value(row))

        def empty_mapper(row):
            yield row

        def reducer(key, rows):
            result = {}
            for row in rows:
                result[row['event_id']] = row
            for row in result.values():
                if row['tag'] == 'old_state_reservation' \
                        or row['tag'] == 'old_state_acceptance' \
                        or row['tag'] == 'old_state_riding' \
                        or row['tag'] == 'old_state_parking':
                    yield row

        if not client.exists(table._get_path()):
            return
        client.create(
            'table',
            path=self._get_path(),
            attributes={
                'schema': self._get_schema(),
            },
            recursive=True,
            ignore_existing=True,
        )
        with client.TempTable(settings.EXPORT['temp_path']) as temp_table:
            temp_path = TablePath(
                name=temp_table,
                append=True,
            )
            client.run_map(
                mapper,
                source_table=table._get_path(),
                destination_table=temp_path,
            )
            client.run_map(
                empty_mapper,
                source_table=self._get_path(),
                destination_table=temp_path,
            )
            client.run_sort(
                source_table=temp_table,
                destination_table=temp_table,
                sort_by=['event_id'],
            )
            client.run_reduce(
                reducer,
                source_table=temp_table,
                destination_table=self._get_path(),
                reduce_by=['event_id'],
            )
