# -*- coding: utf-8 -*-

from os import environ
from datetime import datetime, timedelta
from logging import getLogger
from collections import defaultdict
from json import loads
from multiprocessing.dummy import Pool
from yt.wrapper import TablePath
from cars import settings
from cars.export.yt import Table
from cars.export.api.frontend import FrontendApiClient


# Yt should not fail now
if environ.get('DJANGO_SETTINGS_MODULE') is not None:
    from django.db.models import Q
    from cars.orders.models import CompiledRide
    from cars.cars.models import (
        CarHardwareHead, CarHardwareVega, CarDocument, CarDocumentAssignment,
        CarDocumentHistory, CarDocumentAssignmentHistory,
    )
    from cars.carsharing.models import Car


LOGGER = getLogger(__name__)


class Order(object):
    def __init__(self, *args, **kwargs):
        self.session_id = kwargs['session_id']
        self.user_id = kwargs['user_id']
        self.car_id = kwargs['car_id']
        self.imei = kwargs['imei']
        self.head_id = kwargs['head_id']
        self.geo_id = kwargs.get('geo_id')
        self.created_at = kwargs['created_at']
        self.completed_at = kwargs['completed_at']
        self.total_cost = kwargs['total_cost']
        self.bonus_cost = kwargs['bonus_cost']
        self.discount_cost = kwargs['discount_cost']
        self.discounts = kwargs['discounts']
        self.discount_by_ids = kwargs['discount_by_ids']
        self.tariff_name = kwargs.get('tariff_name')
        self.tariff_price = kwargs.get('tariff_price')
        self.mileage = kwargs['mileage']
        self.raw_route = kwargs.get('raw_route')
        self.route = kwargs.get('route')
        self.ab_points = kwargs['ab_points']
        self.duration_by_steps = kwargs.get('duration_by_steps')
        self.cost_by_steps = kwargs.get('cost_by_steps')
        self.duration_by_types = kwargs['duration_by_types']
        self.cost_by_types = kwargs['cost_by_types']
        self.overrun = kwargs.get('overrun')
        self.overtime = kwargs.get('overtime')
        self.from_scanner = kwargs.get('from_scanner')
        self.free_duration = kwargs.get('free_duration')
        self.priced_duration = kwargs.get('priced_duration')


class Orders(Table):
    ORDERS_DIR = 'data/orders/1d'

    def __init__(self, day):
        self._day = day
        # Preload data
        documents_qs = (
            CarDocumentHistory.objects
            .using(settings.DB_RO_ID)
            .filter(
                Q(history_action='add') &
                (Q(type='car_hardware_vega') |
                Q(type='car_hardware_head'))
            )
            .iterator()
        )
        documents = {}
        for document in documents_qs:
            doc_id = str(document.id)
            data = loads(document.blob)
            if document.type == 'car_hardware_head':
                head_id = data.get('head_id')
                if head_id:
                    documents[doc_id] = ('head', head_id)
            elif document.type == 'car_hardware_vega':
                imei = data.get('imei')
                if imei:
                    documents[doc_id] = ('vega', int(imei))
        assignments_qs = (
            CarDocumentAssignmentHistory.objects
            .using(settings.DB_RO_ID)
            .order_by('history_event_id')
            .iterator()
        )
        last_state = defaultdict(lambda: {
            'head_id': None,
            'imei': None,
        })
        self._documents = defaultdict(list)
        for assignment in assignments_qs:
            doc_id = str(assignment.document_id)
            if doc_id not in documents:
                continue
            car_id = str(assignment.car_id)
            doc_type, doc_data = documents[doc_id]
            if assignment.history_action == 'add':
                if doc_type == 'head':
                    last_state[car_id]['head_id'] = doc_data
                elif doc_type == 'vega':
                    last_state[car_id]['imei'] = doc_data
            elif assignment.history_action == 'remove':
                if doc_type == 'head':
                    last_state[car_id]['head_id'] = None
                elif doc_type == 'vega':
                    last_state[car_id]['imei'] = None
            self._documents[car_id].append((
                assignment.history_timestamp,
                last_state[car_id]
            ))
        self._car_imei = {}
        for car in Car.objects.all():
            if car.imei:
                self._car_imei[str(car.id)] = car.imei
        LOGGER.info('Cars documents are loaded from database!')
        self._api = FrontendApiClient(settings.EXPORT['robot_oauth_token'])

    def _value_to_row(self, value):
        return {
            'session_id': value.session_id,
            'user_id': value.user_id,
            'car_id': value.car_id,
            'imei': value.imei,
            'head_id': value.head_id,
            'geo_id': value.geo_id,
            'created_at': value.created_at,
            'completed_at': value.completed_at,
            'total_cost': value.total_cost,
            'bonus_cost': value.bonus_cost,
            'discount_cost': value.discount_cost,
            'discounts': value.discounts,
            'discount_by_ids': value.discount_by_ids,
            'tariff_name': value.tariff_name,
            'tariff_price': value.tariff_price,
            'mileage': value.mileage,
            'raw_route': value.raw_route,
            'route': value.route,
            'ab_points': value.ab_points,
            'duration_by_steps': value.duration_by_steps,
            'cost_by_steps': value.cost_by_steps,
            'duration_by_types': value.duration_by_types,
            'cost_by_types': value.cost_by_types,
            'overrun': value.overrun,
            'overtime': value.overtime,
            'from_scanner': value.from_scanner,
            'free_duration': value.free_duration,
            'priced_duration': value.priced_duration,
        }

    def _row_to_value(self, row):
        return Order(
            session_id=row['session_id'],
            user_id=row['user_id'],
            car_id=row['car_id'],
            imei=row['imei'],
            head_id=row['head_id'],
            geo_id=row['geo_id'],
            created_at=row['created_at'],
            completed_at=row['completed_at'],
            total_cost=row['total_cost'],
            bonus_cost=row['bonus_cost'],
            discount_cost=row['discount_cost'],
            discounts=row['discounts'],
            discount_by_ids=row['discount_by_ids'],
            tariff_name=row['tariff_name'],
            tariff_price=row['tariff_price'],
            mileage=row['mileage'],
            raw_route=row['raw_route'],
            route=row['route'],
            ab_points=row['ab_points'],
            duration_by_steps=row['duration_by_steps'],
            cost_by_steps=row['cost_by_steps'],
            duration_by_types=row['duration_by_types'],
            cost_by_types=row['cost_by_types'],
            overrun=row['overrun'],
            overtime=row['overtime'],
            from_scanner=row['from_scanner'],
            free_duration=row['free_duration'],
            priced_duration=row['priced_duration'],
        )

    def _get_path(self):
        return '{}/{}/{}'.format(
            settings.EXPORT['home_directory'],
            self.ORDERS_DIR,
            self._day.strftime('%Y-%m-%d'),
        )

    def _get_schema(self):
        return [
            {'name': 'session_id',        'type': 'string',  'required': True},
            {'name': 'user_id',           'type': 'string',  'required': True},
            {'name': 'car_id',            'type': 'string',  'required': True},
            {'name': 'imei',              'type': 'int64'},
            {'name': 'head_id',           'type': 'string'},
            {'name': 'geo_id',            'type': 'string'},
            {'name': 'created_at',        'type': 'int64'},
            {'name': 'completed_at',      'type': 'int64'},
            {'name': 'total_cost',        'type': 'double'},
            {'name': 'bonus_cost',        'type': 'double'},
            {'name': 'discount_cost',     'type': 'double'},
            {'name': 'discounts',         'type': 'any'},
            {'name': 'discount_by_ids',   'type': 'any'},
            {'name': 'tariff_name',       'type': 'string'},
            {'name': 'tariff_price',      'type': 'any'},
            {'name': 'mileage',           'type': 'double'},
            {'name': 'raw_route',         'type': 'any'},
            {'name': 'route',             'type': 'any'},
            {'name': 'ab_points',         'type': 'any'},
            {'name': 'duration_by_steps', 'type': 'any'},
            {'name': 'cost_by_steps',     'type': 'any'},
            {'name': 'duration_by_types', 'type': 'any'},
            {'name': 'cost_by_types',     'type': 'any'},
            {'name': 'overrun',           'type': 'double'},
            {'name': 'overtime',          'type': 'int64'},
            {'name': 'from_scanner',      'type': 'boolean'},
            {'name': 'free_duration',     'type': 'int64'},
            {'name': 'priced_duration',   'type': 'int64'},
        ]

    def save(self, client):
        client.create(
            'table',
            path=self._get_path(),
            attributes={
                'schema': self._get_schema(),
            },
            recursive=True,
            ignore_existing=True,
        )
        client.write_table(
            table=TablePath(
                name=self._get_path(),
                schema=self._get_schema(),
            ),
            input_stream=map(self._value_to_row, self._load_db_data()),
        )

    def _find_document(self, car_id, from_time):
        last_state = {'head_id': None, 'imei': None}
        for document in self._documents[car_id]:
            if document[0] > from_time:
                break
            last_state = document[1]
        if last_state['imei'] is None:
            last_state['imei'] = self._car_imei.get(car_id)
        return last_state['head_id'], last_state['imei']

    def _load_db_data(self):
        begin_time = self._day.replace(
            hour=0,
            minute=0,
            second=0,
            microsecond=0,
        )
        end_time = begin_time + timedelta(days=1)
        rides_qs = (
            CompiledRide.objects
            .using(settings.DB_RO_ID)
            .filter(
                finish__gte=int(begin_time.timestamp()),
                finish__lt=int(end_time.timestamp()),
            )
        )
        for ride in rides_qs.iterator():
            if ride.meta_proto is None:
                yield self._parse_ride_with_old_meta(ride)
            else:
                yield self._parse_ride(ride)

    def _parse_ride(self, ride):
        total_cost = float()
        bonus_cost = float()
        discount_cost = float()
        discounts = []
        discount_by_ids = defaultdict(float)
        duration_by_types = defaultdict(int)
        cost_by_types = defaultdict(float)
        created_at = ride.start
        completed_at = ride.finish
        free_duration = 0
        priced_duration = completed_at - created_at
        meta = ride.get_meta()
        if meta.Bill:
            free_duration = meta.Bill.FreeDuration
            priced_duration = meta.Bill.PricedDuration
            for record in meta.Bill.Record:
                if record.Type == 'total':
                    total_cost = abs(float(record.Cost)) / 100
                elif record.Type == 'discount':
                    record_cost = abs(float(record.Cost)) / 100
                    discounts.append({
                        'id': str(record.Id),
                        'title': record.Title,
                        'cost': record_cost,
                        'details': record.Details,
                    })
                    discount_cost += record_cost
                    discount_by_ids[record.Id] += record_cost
                elif record.Type == 'billing_bonus':
                    bonus_cost += abs(float(record.Cost)) / 100
                if record.Duration:
                    duration_by_types[record.Type] += record.Duration
                if record.Cost:
                    record_cost = abs(float(record.Cost)) / 100
                    if record.Type == 'discount' \
                            or record.Type == 'billing_bonus':
                        record_cost = -record_cost
                    if record.Type != 'total':
                        cost_by_types[record.Type] += record_cost
        ab_points = {
            'start': None,
            'finish': None,
        }
        mileage = float()
        if meta.SnapshotsDiff:
            diff = meta.SnapshotsDiff
            mileage = float(diff.Mileage)
            if diff.Start:
                ab_points['start'] = (
                    diff.Start.Latitude,
                    diff.Start.Longitude,

                )
            if diff.Last:
                ab_points['finish'] = (
                    diff.Last.Latitude,
                    diff.Last.Longitude,
                )
        session_id = str(ride.session_id)
        car_id = str(ride.object_id)
        head_id, imei = self._find_document(car_id, created_at)
        return Order(
            session_id=session_id,
            user_id=self._to_string(ride.history_user_id),
            car_id=car_id,
            imei=imei,
            head_id=head_id,
            created_at=created_at,
            completed_at=completed_at,
            total_cost=total_cost,
            bonus_cost=bonus_cost,
            discount_cost=discount_cost,
            discounts=discounts,
            discount_by_ids=discount_by_ids,
            mileage=mileage,
            ab_points=ab_points,
            duration_by_types=duration_by_types,
            cost_by_types=cost_by_types,
            free_duration=free_duration,
            priced_duration=priced_duration,
        )

    def _parse_ride_with_old_meta(self, ride):
        total_cost = float()
        bonus_cost = float()
        discount_cost = float()
        discounts = []
        discount_by_ids = defaultdict(float)
        duration_by_types = defaultdict(int)
        cost_by_types = defaultdict(float)
        created_at = ride.start
        completed_at = ride.finish
        free_duration = 0
        priced_duration = completed_at - created_at
        if ride.meta and 'bill' in ride.meta:
            bill = ride.meta['bill']
            records = bill
            if isinstance(bill, dict):
                records = bill.get('records', [])
            if 'free_duration' in bill:
                free_duration = bill['free_duration']
            if 'priced_duration' in bill:
                priced_duration = bill['priced_duration']
            for record in records:
                record_type = record.get('type')
                if record_type == 'total':
                    total_cost = abs(float(record.get('cost', 0))) / 100
                elif record_type == 'discount':
                    record_cost = abs(float(record.get('cost', 0))) / 100
                    discounts.append({
                        'id': str(record.get('id')),
                        'title': record.get('title'),
                        'cost': record_cost,
                        'details': record.get('details'),
                    })
                    discount_cost += record_cost
                    discount_by_ids[record.get('id')] += record_cost
                elif record_type == 'billing_bonus':
                    record_cost = abs(float(record.get('cost', 0))) / 100
                    bonus_cost += record_cost
                if record.get('duration'):
                    duration_by_types[record['type']] += record['duration']
                if 'cost' in record:
                    record_cost = abs(float(record['cost'])) / 100
                    if record_type == 'discount' or record_type == 'billing_bonus':
                        record_cost = -record_cost
                    cost_by_types[record['type']] += record_cost
        ab_points = {
            'start': None,
            'finish': None,
        }
        mileage = float()
        if ride.meta and 'diff' in ride.meta:
            diff = ride.meta['diff']
            mileage = float(diff.get('mileage', 0))
            if 'start' in diff:
                ab_points['start'] = self._to_geo_point(diff['start'])
            if 'finish' in diff:
                ab_points['finish'] = self._to_geo_point(diff['finish'])
        session_id = str(ride.session_id)
        car_id = str(ride.object_id)
        head_id, imei = self._find_document(car_id, created_at)
        return Order(
            session_id=session_id,
            user_id=self._to_string(ride.history_user_id),
            car_id=car_id,
            imei=imei,
            head_id=head_id,
            created_at=created_at,
            completed_at=completed_at,
            total_cost=total_cost,
            bonus_cost=bonus_cost,
            discount_cost=discount_cost,
            discounts=discounts,
            discount_by_ids=discount_by_ids,
            mileage=mileage,
            ab_points=ab_points,
            duration_by_types=duration_by_types,
            cost_by_types=cost_by_types,
            free_duration=free_duration,
            priced_duration=priced_duration,
        )

    @staticmethod
    def _to_string(value):
        return str(value) if value else None

    @staticmethod
    def _to_integer(value):
        return int(value) if value else None

    @staticmethod
    def _to_geo_point(value):
        return (
            value['latitude'],
            value['longitude']
        )
