# -*- coding: utf-8 -*-

from os import environ
from datetime import datetime, timedelta
from logging import getLogger
from collections import defaultdict
from multiprocessing.dummy import Pool
from yt.wrapper import TablePath
from cars import settings
from cars.export.yt import Table
from cars.export.api.frontend import FrontendApiClient


# Yt should not fail now
if environ.get('DJANGO_SETTINGS_MODULE') is not None:
    from cars.orders.models import CompiledRide


LOGGER = getLogger(__name__)


class Route(object):
    def __init__(self, *args, **kwargs):
        self.session_id = kwargs['session_id']
        self.raw_route = kwargs.get('raw_route')
        self.route = kwargs.get('route')


class Routes(Table):
    ROUTES_DIR = 'data/orders/_helpers/routes'

    def __init__(self, day):
        self._day = day
        self._api = FrontendApiClient(settings.EXPORT['robot_oauth_token'])

    def _value_to_row(self, value):
        return {
            'session_id': value.session_id,
            'raw_route': value.raw_route,
            'route': value.route,
        }

    def _row_to_value(self, row):
        return Order(
            session_id=row['session_id'],
            raw_route=row['raw_route'],
            route=row['route'],
        )

    def _get_path(self):
        return '{}/{}/{}'.format(
            settings.EXPORT['home_directory'],
            self.ROUTES_DIR,
            self._day.strftime('%Y-%m-%d'),
        )

    def _get_schema(self):
        return [
            {'name': 'session_id', 'type': 'string',  'required': True},
            {'name': 'raw_route',  'type': 'any'},
            {'name': 'route',      'type': 'any'},
        ]

    def save(self, client, complete):
        client.create(
            'table',
            path=self._get_path(),
            attributes={
                'schema': self._get_schema(),
            },
            recursive=True,
            ignore_existing=True,
        )
        if not client.get_attribute(self._get_path(), '_complete', False):
            client.write_table(
                table=TablePath(
                    name=self._get_path(),
                    schema=self._get_schema(),
                ),
                input_stream=map(self._value_to_row, self._load_data()),
            )
            client.set_attribute(self._get_path(), '_complete', complete)

    def _load_data(self):
        pool = Pool(48)
        orders = pool.map(self._load_api_data, self._get_session_ids())
        pool.close()
        pool.join()
        return filter(None.__ne__, orders)

    def _load_api_data(self, session_id):
        try:
            raw_route = self._api.get_session_raw_route(session_id)
            route = self._api.get_session_route(session_id)
            return Route(
                session_id=session_id,
                raw_route=raw_route,
                route=route,
            )
        except RuntimeError:
            LOGGER.error('Unable to load route')
            return None

    def _get_session_ids(self):
        begin_time = self._day.replace(
            hour=0,
            minute=0,
            second=0,
            microsecond=0,
        )
        end_time = begin_time + timedelta(days=1)
        return (
            CompiledRide.objects
            .using(settings.DB_RO_ID)
            .filter(
                finish__gte=int(begin_time.timestamp()),
                finish__lt=int(end_time.timestamp()),
            )
            .values_list('session_id', flat=True)
        )
