# -*- coding: utf-8 -*-

from os import environ
from logging import getLogger
from collections import defaultdict
from yt.wrapper import TablePath
from cars import settings
from cars.export.yt import Table



# Yt should not fail now
if environ.get('DJANGO_SETTINGS_MODULE') is not None:
    from cars.users.models import UserTag


LOGGER = getLogger(__name__)


class UserTagsRow(object):
    def __init__(self, *args, **kwargs):
        self.user_id = kwargs['user_id']
        self.tags = kwargs.get('tags')


class UserTags(Table):
    def _value_to_row(self, value):
        return {
            'id': value.user_id,
            'user_tags': value.tags,
        }

    def _row_to_value(self, row):
        return Order(
            user_id=row['id'],
            tags=row['user_tags'],
        )

    def _get_path(self):
        return '{}/{}'.format(
            settings.EXPORT['home_directory'],
            'data/user/_parts/user_tags',
        )

    def _get_schema(self):
        return [
            {'name': 'id',        'type': 'string',  'required': True},
            {'name': 'user_tags', 'type': 'any'                      },
        ]

    def save(self, client):
        client.create(
            'table',
            path=self._get_path(),
            attributes={
                'schema': self._get_schema(),
            },
            recursive=True,
            ignore_existing=True,
        )
        with client.Transaction():
            client.write_table(
                table=TablePath(
                    name=self._get_path(),
                    schema=self._get_schema(),
                ),
                input_stream=map(self._value_to_row, self._load_data()),
            )
            client.run_sort(
                source_table=self._get_path(),
                destination_table=self._get_path(),
                sort_by=['id'],
            )

    def _load_data(self):
        user_tags = defaultdict(list)
        for user_tag in UserTag.objects.all():
            if not user_tag.object_id:
                continue
            user_tags[str(user_tag.object_id)].append(user_tag.tag)
        for user_id, tags in user_tags.items():
            yield UserTagsRow(
                user_id=user_id,
                tags=tags,
            )
