from . import base
from .base import entities
from . import utils


class Resource(base.Resource):
    def __init__(self, api_root):
        super(Resource, self).__init__(
            uri=api_root.assignments,
            collection_class=Collection,
            item_class=Item,
            entity_class=Entity,
        )


class Collection(base.Collection):
    pass


class Item(base.Item):
    def accept(self, public_comment):
        """ Accepts current assignment. """
        return base.Entity(
            self.uri.PATCH(json={'status': 'ACCEPTED', 'public_comment': public_comment}).json()
        )

    def reject(self, public_comment):
        """ Rejects current assignment. """
        return base.Entity(
            self.uri.PATCH(json={'status': 'REJECTED', 'public_comment': public_comment}).json()
        )


class Entity(entities.Datable, base.Entity):
    """ An entity for assignment.

    Attributes:
        id (str): An identifier.
        task_suite_id (str): Related :obj:`.TaskSuite` identifier.
        pool_id (str): Related :obj:`.Pool` identifier.
        user_id (str): An :obj:`.User` identifier.
        stutus (str): An assignment's status.
        created (:obj:`datetime`): An assignment's start time.
        submitted (:obj:`datetime`): An assignment's submission time.
        accepted (:obj:`datetime`): An assignment's acceptance time.
        tasks (:obj:`list` of `dict`): An list of input values mapping.
        solutions (:obj:`list` of `dict`): An list of output values mapping.

    .. _Assignment:
            https://tech.yandex.ru/toloka/api/doc/concepts/result-docpage/
    """

    DATE_FIELDS = [
        'created',
        'submitted',
        'accepted',
    ]

    @property
    def input_values(self):
        """ :obj:`list` of :obj:`dict`: A list of input values (from `input_spec`). """
        return (s['input_values'] for s in self['tasks'])

    @property
    def output_values(self):
        """ :obj:`list` of :obj:`dict`: A list of output values (from `output_spec`). """
        return (s['output_values'] for s in self['solutions'])

    @property
    def micro_tasks(self):
        """ iterable of :obj:`dict`: An iterator over merged `input` and `output` values. """
        return map(utils.merge_dicts, self.input_values, self.output_values)
