from attrdict import AttrDict

from .. import utils


class Entity(AttrDict):
    # Makes subdictionaries AttrDict instances, not Entity
    _constructor = AttrDict._constructor


class Item(object):
    """ An item
    Allows to get entity or perform operation on it
    """

    def __init__(self, resource, id, entity_class):
        self.uri = resource.uri(id)
        self.entity_class = entity_class

    def get(self):
        """ Retrieves an object. """
        return self.entity_class(
            **self.uri.GET().json()
        )


class Collection(object):
    """ A collection of items
    Allows to either get all items or create a new one
    """
    MAX_LIMIT = 300

    def __init__(self, resource, entity_class, **kwargs):
        self.uri = resource.uri
        self.entity_class = entity_class
        self.params = utils.prepare_params(kwargs)

    def __iter__(self):
        return (
            self.entity_class(uri=self.uri(data['id']), **data)
            for data in utils.cursor(self.uri, params=self.params)
        )

    def __getitem__(self, item):
        if isinstance(item, slice):
            raise NotImplementedError('slices are not supported yet')
        else:
            params = self.params.copy()
            params.setdefault('limit', min(self.MAX_LIMIT, item+1))
            for i, data in enumerate(utils.cursor(self.uri, params=params)):
                if i == item:
                    break
            else:
                raise IndexError()
            return self.entity_class(uri=self.uri(data['id']), **data)


class Resource(object):
    """ A common class for working with Toloka resources. """

    def __init__(self, uri, collection_class=None, item_class=None, entity_class=Entity):
        """ Initializes a new resource.

        Attributes:
            uri (:obj:`toloka.karrimat.Karrimat`): An api root for making requests.
            entity_class (:class:`.TolokaEntity`): A class to manage.
        """
        self.collection_class = collection_class
        self.item_class = item_class
        self.entity_class = entity_class
        self.uri = uri

    def __iter__(self):
        """ Allows use this objects as iterator. """
        return iter(self.collection_class(self, self.entity_class))

    def __getitem__(self, item):
        return self.collection_class(self, self.entity_class)[item]

    def __getattr__(self, item):
        return getattr(self.collection_class(self, self.entity_class), item)

    def __call__(self, id=None, **kwargs):
        """ Used to perform interaction with object(s).

        Args:
            id:
            kwargs: Additional query parameters to pass in request (e.g. `id_gt`, `created_lt`).
        """
        if id:
            return self.item_class(self, id, self.entity_class)
        else:
            return self.collection_class(self, self.entity_class, **kwargs)
