import datetime

DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%S.%f'


def dates_to_python(data, date_fields):
    """Converts date strings to python datetime objects."""
    for field in date_fields:
        if field in data:
            ts = data[field]
            # if microseconds is not present, fill them with 0
            if '.' not in ts:
                ts += '.0'
            data[field] = datetime.datetime.strptime(ts, DATETIME_FORMAT)

    return data


def cursor(url, params=None):
    """ Allows easily iterate over items available by URL.

    This functions deals with Toloka's `has_more` and `items` fields of GET results and
    transforms them to an items iterator.

    Args:
        url (:obj:`.base.Resource`): API's url for item retrieving.
        params (:obj:`dict`, optional): Optional params to pass in GET request.

    Yields:
        dict: The next item.
    """
    if params:
        params = params.copy()
    else:
        params = {}

    params.setdefault('limit', 100)
    params.setdefault('sort', 'created')

    if params['sort'].startswith('-'):
        pagination_param_name = params['sort'][1:]
        pagination_filtering_param_name = pagination_param_name + '_lt'
    else:
        pagination_param_name = params['sort']
        pagination_filtering_param_name = pagination_param_name + '_gt'
    has_more = True

    while has_more:
        resp = url.GET(params=params).json()
        for item in resp['items']:
            yield item
        has_more = resp['has_more']
        if has_more and item:
            params[pagination_filtering_param_name] = item[pagination_param_name]


def prepare_params(data):
    for key, value in data.items():
        if isinstance(value, datetime.datetime):
            data[key] = value.strftime(DATETIME_FORMAT)
        elif isinstance(value, bool):
            data[key] = str(value).lower()
    return data


def prepare_json(data):
    for key, value in data.items():
        if isinstance(value, datetime.datetime):
            data[key] = value.strftime(DATETIME_FORMAT)
    return data


def merge_dicts(d1, d2):
    """ Given two dicts, merge them into a new dict as a shallow copy. """
    result = d1.copy()
    result.update(d2)
    return result
