# pylint: disable=line-too-long
import random
import time

from django.db import transaction
from django.shortcuts import get_object_or_404
from django.utils import timezone
from rest_framework.permissions import IsAuthenticated
from rest_framework.response import Response

from cars.django.views import CarsharingAPIView
from .models import FakeCar, FakeOrder
from .serializers import FakeCarSerializer, FakeOrderSerializer


class FakeView(CarsharingAPIView):
    permission_classes = [IsAuthenticated]


class CarListFakeView(FakeView):

    def do_get(self, request):
        _ = request
        cars = FakeCar.objects.filter(is_free=True)
        data = FakeCarSerializer(cars, many=True).data
        return Response({
            'cars': data,
        })


class CarModelsFakeView(FakeView):

    def do_get(self, request):
        _ = request
        return Response([
            {
                'id': 1,
                'fuel_type': 'petrol',
                'image_url': 'https://carsharing-api.yandex.net/aggregator_testing/static/model-images/solaris.png?rev=3',
                'name': 'Hyundai Solaris',
                'seats': 4,
                'short_name': 'solaris'
            },
            {
                'id': 2,
                'fuel_type': 'petrol',
                'image_url': 'https://carsharing-api.yandex.net/aggregator_testing/static/model-images/kia_rio.png?rev=3',
                'name': 'KIA Rio',
                'seats': 4,
                'short_name': 'kia_rio'
            },
            {
                'id': 3,
                'fuel_type': 'petrol',
                'image_url': 'https://carsharing-api.yandex.net/aggregator_testing/static/model-images/ford_fiesta.png?rev=3',
                'name': 'Ford Fiesta',
                'seats': 4,
                'short_name': 'ford_fiesta'
            },
        ])


class RentCarFakeView(FakeView):

    def do_post(self, request, car_id):
        car = get_object_or_404(FakeCar, id=car_id)

        order = (
            FakeOrder.objects
            .filter(
                user=request.user,
                car=car,
                finished_at__isnull=True
            )
            .first()
        )
        if order is not None:
            return Response()

        car.is_free = False
        car.updated_at = timezone.now()
        car.update_timestamp = time.time()

        with transaction.atomic():
            FakeOrder.objects.create(
                user=request.user,
                car=car,
                started_at=timezone.now(),
                finished_at=None,
                costs=[],
            )
            car.save()

        return Response()


class FreeCarFakeView(FakeView):

    def do_post(self, request, car_id):
        car = get_object_or_404(FakeCar, id=car_id)
        order = get_object_or_404(FakeOrder, user=request.user, car=car, finished_at__isnull=True)

        order.finished_at = timezone.now()
        cost = {
            'cost' : 1 * (order.finished_at - order.started_at).seconds / 60,
            'trip_id' : random.randrange(0, 100000),
            'id' : random.randrange(0, 100000),
            'type' : 'waiting'
        }
        order.costs.append(cost)
        car.is_free = True
        car.updated_at = timezone.now()

        with transaction.atomic():
            order.save()
            car.save()

        return Response({
            'order': FakeOrderSerializer(order).data,
        })


class CurrentOrderFakeView(FakeView):

    def do_get(self, request):
        order = (
            FakeOrder.objects
            .filter(
                user=request.user,
                finished_at__isnull=True,
            )
            .order_by('-started_at')
            .first()
        )

        if order is None:
            data = None
        else:
            data = FakeOrderSerializer(order).data

        return Response({
            'order': data,
        })


class FakeOrderListView(FakeView):

    def do_get(self, request):
        orders = FakeOrder.objects.filter(user=request.user).order_by('-started_at')
        return Response({
            'orders': FakeOrderSerializer(orders, many=True).data,
        })
