# coding: utf-8
from __future__ import unicode_literals
import json
import logging
import math

import saaspy

from cars.aggregator.static_data import car_models
from cars.aggregator.static_data.operators import OPERATORS
from cars.core.models.car_model import CarModel
from cars.core.util import get_city


LOGGER = logging.getLogger(__name__)


class BadSaasDocumentError(Exception):
    pass


class Car(object):
    '''Base class to be extended in operator implementations.'''

    # Cars may report fluctuating GPS positions while being free.
    # Applying a threshold to determine whether a free car changed its position
    #   avoids marking the car as updated on minor GPS fluctuations.
    MOVING_THRESHOLD_WHEN_FREE = 0.002

    COLOR = CarModel.Color
    _COLOR_MAP = {}

    TRANSMISSION = CarModel.Transmission
    _TRANSMISSION_MAP = {}

    def __init__(self, local_id, city_id, operator, color,
                 discount, fuel, lat, lon, address_en, address_ru, model,
                 plate_number, parking_tariff, usage_tariff, transmission, is_free, updated_at):
        self.local_id = local_id
        self.city_id = city_id
        self.operator = operator
        self.color = color
        self.discount = discount
        self.fuel = fuel
        self.lat = lat
        self.lon = lon
        self.address_en = address_en
        self.address_ru = address_ru
        self.model = model
        self.plate_number = plate_number
        self.parking_tariff = parking_tariff
        self.usage_tariff = usage_tariff
        self.transmission = transmission
        self.is_free = is_free
        self.updated_at = updated_at

    def __eq__(self, other):
        if self.is_free and other.is_free:
            is_lat_changed = math.fabs(self.lat - other.lat) > self.MOVING_THRESHOLD_WHEN_FREE
            is_lon_changed = math.fabs(self.lon - other.lon) > self.MOVING_THRESHOLD_WHEN_FREE
            is_pos_changed = is_lat_changed or is_lon_changed
        else:
            is_pos_changed = self.lat != other.lat or self.lon != other.lon

        return (
            self.id == other.id
            and not is_pos_changed
            and self.fuel == other.fuel
            and self.is_free == other.is_free
            and self.usage_tariff == other.usage_tariff
            and self.parking_tariff == other.parking_tariff
            and self.discount == other.discount
            and self.city_id == other.city_id
            and self.model == other.model
        )

    def __ne__(self, other):
        return not self == other

    @property
    def id(self):
        '''A unique id within all operators.'''
        return '{}-{}'.format(self.operator.short_name, self.local_id)

    @property
    def position(self):
        return {
            'lat': self.lat,
            'lon': self.lon,
        }

    @property
    def tariff(self):
        return {
            'parking': self.parking_tariff,
            'usage': self.usage_tariff,
            'discount': self.discount,
        }

    @classmethod
    def from_raw(cls, operator, raw_data, is_free, updated_at):
        plate_number = cls.extract_plate_number(raw_data)
        if plate_number is not None:
            plate_number = cls.format_plate_number(plate_number)

        # Some cars are missing fuel level info.
        # It's better to display 0 than to skip the car.
        try:
            fuel = cls.extract_fuel(raw_data)
        except Exception:
            fuel = 0.0

        return cls(
            operator=operator,
            local_id=cls.extract_local_id(raw_data),
            city_id=cls.extract_city_id(raw_data),
            color=cls.extract_color(raw_data),
            discount=cls.extract_discount(raw_data),
            fuel=fuel,
            lat=cls.extract_lat(raw_data),
            lon=cls.extract_lon(raw_data),
            address_en='',
            address_ru='',
            model=cls.extract_normalized_model(raw_data),
            plate_number=plate_number,
            parking_tariff=cls.extract_parking_tariff(raw_data),
            usage_tariff=cls.extract_usage_tariff(raw_data),
            transmission=cls.extract_transmission(raw_data),
            is_free=is_free,
            updated_at=updated_at,
        )

    @classmethod
    def extract_local_id(cls, raw_data):
        '''A unique id within an operator.'''
        raise NotImplementedError

    @classmethod
    def extract_city_id(cls, raw_data):
        lat = cls.extract_lat(raw_data)
        lon = cls.extract_lon(raw_data)
        city = get_city(lat=lat, lon=lon)
        return city.code if city else None

    @classmethod
    def extract_color(cls, raw_data):
        raise NotImplementedError

    @classmethod
    def extract_discount(cls, _):
        return 0.0

    @classmethod
    def extract_fuel(cls, raw_data):
        raise NotImplementedError

    @classmethod
    def extract_lat(cls, raw_data):
        raise NotImplementedError

    @classmethod
    def extract_lon(cls, raw_data):
        raise NotImplementedError

    @classmethod
    def extract_normalized_model(cls, raw_data):
        raw_model = cls.extract_model(raw_data).lower()
        if raw_model == 'bmw 116i':
            model = car_models.BMW_116I
        elif raw_model == 'chevrolet aveo':
            model = car_models.CHEVROLET_AVEO
        elif raw_model == 'spark':
            model = car_models.CHEVROLET_SPARK
        elif raw_model == 'datsun mi-do':
            model = car_models.DATSUN
        elif raw_model == 'fiat 500':
            model = car_models.FIAT_500
        elif raw_model == 'ford fiesta':
            model = car_models.FORD_FIESTA
        elif raw_model in ('focus', 'focus w'):
            model = car_models.FORD_FOCUS
        elif raw_model in ('hyundai solaris', 'solaris'):
            model = car_models.SOLARIS
        elif raw_model == 'solaris new':
            model = car_models.SOLARIS_NEW
        elif raw_model in ('kia rio', 'kia_rio'):
            model = car_models.KIA_RIO
        elif raw_model == 'kia_rio_xline':
            model = car_models.KIA_RIO_X_LINE
        elif raw_model == 'mercedes-benz cla':
            model = car_models.MERCEDES_BENZ
        elif raw_model == 'mini cooper':
            model = car_models.MINI_COOPER
        elif raw_model in ('almera', 'nissan almera'):
            model = car_models.NISSAN_ALMERA
        elif raw_model in ('r2', 'r-2') or 'ravon r2' in raw_model:
            # DriveTime has Ravon R2 model name formatted as '(Ravon R2 )'
            model = car_models.RAVON_R2
        elif raw_model == 'ravon r3':
            model = car_models.RAVON_R3
        elif raw_model == 'renault kangoo':
            model = car_models.RENAULT_KANGOO
        elif raw_model in ('renault kaptur', 'renault_kaptur'):
            model = car_models.RENAULT_KAPTUR
        elif raw_model == 'renault logan':
            model = car_models.RENAULT_LOGAN
        elif raw_model in ('renault stepway', 'sandero stepway'):
            model = car_models.RENAULT_SANDERO
        elif raw_model == 'skoda fabia':
            model = car_models.SKODA_FABIA
        elif raw_model == 'skoda octavia':
            model = car_models.SKODA_OCTAVIA
        elif raw_model in ('rapid', 'skoda rapid'):
            model = car_models.SKODA_RAPID
        elif raw_model in ('smart forfour', 'forfour turbo'):
            model = car_models.SMART_FORFOUR
        elif raw_model in ('smart fortwo', 'fortwo turbo'):
            model = car_models.SMART_FORTWO
        elif raw_model == 'vw e-up!':
            model = car_models.VOLKSWAGEN_E_UP
        elif raw_model in ('vw polo', 'volkswagen polo'):
            model = car_models.VOLKSWAGEN_POLO
        elif raw_model == 'opel astra':
            model = car_models.OPEL_ASTRA
        else:
            model = car_models.DEFAULT_MODEL
            LOGGER.warning('Unknown model: %s', raw_model)
        return model.short_name

    @classmethod
    def extract_model(cls, raw_data):
        raise NotImplementedError

    @classmethod
    def extract_plate_number(cls, raw_data):
        raise NotImplementedError

    @classmethod
    def format_plate_number(cls, plate_number):
        plate_number = plate_number.upper()
        if len(plate_number) != 9:
            return plate_number

        series1 = plate_number[0]
        reg_number = plate_number[1:4]
        series2 = plate_number[4:6]
        region = plate_number[6:9]

        formatted_plate_number = u'{} {} {} {}'.format(series1, reg_number, series2, region)

        return formatted_plate_number

    @classmethod
    def extract_parking_tariff(cls, raw_data):
        raise NotImplementedError

    @classmethod
    def extract_usage_tariff(cls, raw_data):
        raise NotImplementedError

    @classmethod
    def extract_transmission(cls, raw_data):
        raise NotImplementedError

    @classmethod
    def _map_color(cls, raw_color):
        if raw_color:
            raw_color = cls._normalize_color(raw_color)

        if raw_color in cls._COLOR_MAP:
            color = cls._COLOR_MAP[raw_color].value
        elif raw_color:
            color = raw_color.capitalize()
        else:
            color = None

        return color

    @classmethod
    def _normalize_color(cls, color):
        color = color.lower()
        color = color.replace('ё', 'е')
        return color

    @classmethod
    def _map_transmission(cls, raw_transmission):
        if raw_transmission in cls._TRANSMISSION_MAP:
            transmission = cls._TRANSMISSION_MAP[raw_transmission].value
        else:
            transmission = raw_transmission
        return transmission

    @classmethod
    def from_saas_doc(cls, doc):
        try:
            is_free = int(doc['is_free']) != 0
            city_id = int(doc['city_id'])
            operator = OPERATORS[doc['operator']]
            updated_at = int(doc['updated_at'])
            data = json.loads(doc['data'])
        except KeyError:
            raise BadSaasDocumentError(doc.url)

        return cls(
            operator=operator,
            local_id=data['local_id'],
            city_id=city_id,
            color=data['color'],
            fuel=data['fuel'],
            lat=data['position']['lat'],
            lon=data['position']['lon'],
            address_en=data.get('address_en', data.get('address')),
            address_ru=data.get('address_ru', data.get('address')),
            model=data['model'],
            plate_number=data['plate_number'],
            discount=data.get('tariff', {}).get('discount', 0.0),
            parking_tariff=data.get('tariff', {}).get('parking', 0.0),
            usage_tariff=data.get('tariff', {}).get('usage', 0.0),
            transmission=data['transmission'],
            is_free=is_free,
            updated_at=updated_at,
        )

    def to_dict(self):
        data = {
            'id': self.id,
            'local_id': self.local_id,
            'city_id': self.city_id,
            'address_en': self.address_en,
            'address_ru': self.address_ru,
            'color': self.color,
            'fuel': self.fuel,
            'model': self.model,
            'operator': self.operator.short_name,
            'plate_number': self.plate_number,
            'position': self.position,
            'tariff': self.tariff,
            'transmission': self.transmission,
        }
        return data

    def to_saas_doc(self):
        data = self.to_dict()

        # No need for these fields since they are stored as doc properties.
        data.pop('city_id')
        data.pop('operator')

        json_data = json.dumps(data, ensure_ascii=False).encode('utf8')

        doc = saaspy.SaasDocument(self.id)
        doc.add_property('data', json_data)
        doc.add_entity('city_id', self.city_id, property=True, search_attr_int=True)
        doc.add_entity('is_free', 1 if self.is_free else 0, property=True, search_attr_int=True)
        doc.add_entity(
            'operator',
            self.operator.short_name,
            property=True,
            search_attr_literal=True,
        )
        doc.add_entity('updated_at', self.updated_at, property=True, search_attr_int=True)

        return doc
