from __future__ import unicode_literals
import logging
import time

from kubiki.geocoder import Geocoder

from .car import BadSaasDocumentError


LOGGER = logging.getLogger(__name__)


class CarIndex(object):

    FULL_SYNC_INTERVAL = 5 * 60  # Once in 5min.

    def __init__(self, saas_client, operator, car_class, geocoder_url):
        self._geocoder = Geocoder(url=geocoder_url)
        self._saas = saas_client
        self._operator = operator
        self._car_class = car_class

        # id -> Car
        self._cars = {}
        self._full_synced_at = None

    def maybe_full_sync(self):
        now_ts = time.time()
        if self._full_synced_at is None or now_ts - self._full_synced_at > self.FULL_SYNC_INTERVAL:
            cars = self.fetch_all_cars()
            self._cars = {car.id: car for car in cars}
            self._full_synced_at = now_ts

    def update(self, new_free_cars):
        self.maybe_full_sync()

        cars_to_update = []
        new_free_cars = {car.id: car for car in new_free_cars}

        # Load missing cars from the index.
        for car_id in new_free_cars:
            if car_id not in self._cars:
                old_car = self.fetch_car(car_id)
                if old_car:
                    self._cars[car_id] = old_car

        # Collect modified cars.
        for car_id, old_car in self._cars.items():
            new_free_car = new_free_cars.get(car_id)
            if new_free_car:
                if new_free_car != old_car:
                    self._cars[car_id] = new_free_car
                    cars_to_update.append(new_free_car)
            elif old_car.is_free:
                old_car.is_free = False
                cars_to_update.append(old_car)

        # Collect new cars.
        for car_id, new_free_car in new_free_cars.items():
            if car_id not in self._cars:
                self._cars[car_id] = new_free_car
                cars_to_update.append(new_free_car)

        for car in cars_to_update:
            try:
                self.update_car_address(car)
            except Exception:
                LOGGER.exception('Failed to update car address')
                continue

        now_ts = time.time()
        for car in cars_to_update:
            car.updated_at = now_ts

        car_docs = [car.to_saas_doc() for car in cars_to_update]

        return self._saas.modify_many_documents(car_docs, realtime=True)

    def update_car_address(self, car):
        response = self._geocoder.geocode_coords(lat=car.lat, lon=car.lon, lang='en', n_retries=3)
        car.address_en = response.objects[0].name
        response = self._geocoder.geocode_coords(lat=car.lat, lon=car.lon, lang='ru', n_retries=3)
        car.address_ru = response.objects[0].name

    def fetch_car(self, car_id):
        query = self._saas.construct_complex_query({
            'url': '"{}"'.format(car_id),
        })
        response = self._saas.search_by_text(query)
        response.raise_for_status()

        if response.is_empty:
            car = None
        else:
            doc = response.documents[0]
            car = self._parse_car_doc(doc)

        return car

    def fetch_all_cars(self):
        query = self._saas.construct_complex_query({
            'operator': '"{}"'.format(self._operator.short_name),
        })
        extra_params = {
            'numdoc': 10000,
            'balancertimeout': 1000,  # Selecting all cars is a heavy operation.
            'snip': 'diqm=1',  # Disable snippet construction to speedup report.
            'relev': 'attr_limit=99999999',  # Enable more that 1000 responses.
        }
        response = self._saas.search_by_text(query, extra_params=extra_params)
        response.raise_for_status()

        cars = []
        for doc in response.documents:
            car = self._parse_car_doc(doc)
            if car is None:
                continue
            cars.append(car)

        return cars

    def _parse_car_doc(self, doc):
        try:
            car = self._car_class.from_saas_doc(doc)
        except BadSaasDocumentError:
            LOGGER.error('Removing invalid document from index: %s\n%s', doc.url, doc.to_json())
            self._saas.delete_document_by_url(doc.url)
            car = None
        except Exception:
            LOGGER.exception('Failed to parse car document:\n%s', doc)
            car = None
        return car
