# pylint: disable=redefined-outer-name
import logging
import re
import time

import kubiki
import requests
import saaspy
from planex.base import DistributedService
from planex.monitoring import DummyServiceMonitor, SolomonServiceMonitor

from .index import CarIndex
from .push_client import FeedsPushClientLogger


LOGGER = logging.getLogger(__name__)


class FeedServiceMonitor(SolomonServiceMonitor):

    def _make_service_prefix(self, service_name):
        match = re.match(r'(\w+)Feed', service_name)
        if not match:
            raise RuntimeError(
                '{} does not comply feed delivery naming scheme'.format(service_name)
            )
        feed_name = match.group(1)
        return '{}_feed'.format(feed_name.lower())

    def _make_extra(self, *args, **kwargs):  # pylint: disable=arguments-differ
        extra = super()._make_extra(*args, **kwargs)
        extra['labels']['type'] = 'feed'
        return extra


class FeedMeta(type):
    '''Automatically add Feed subclasses to the registry'''

    def __init__(cls, *args, **kwargs):
        super().__init__(*args, **kwargs)
        cls.registry[cls.operator] = cls


class Feed(DistributedService, metaclass=FeedMeta):

    registry = {}  # Operator -> Feed registry containing all Feed subclasses.

    operator = None
    CarClass = None
    ConfigClass = None

    tick_interval = '* * * * * *'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._saas = saaspy.SaasClient(**self.cfg.saas)
        self._index = CarIndex(
            self._saas, self.cfg.operator, self.CarClass, self.cfg.geocoder_url,
        )
        self._push_client = FeedsPushClientLogger(self.cfg.push_client_filename)
        self._session = kubiki.util.make_requests_session()

    @property
    def request_url(self):
        raise NotImplementedError

    @property
    def request_headers(self):
        return {}

    @property
    def request_timeout(self):
        return 10

    def get_configs(self):
        return self.ConfigClass(operator=self.operator)  # pylint: disable=not-callable

    def _extract_feed(self, response):
        raise NotImplementedError

    @property
    def logging_config(self):
        cfg = super().logging_config
        cfg['loggers']['requests.packages.urllib3'] = {
            'level': 'DEBUG',
        }
        if self.cfg.sentry_dsn:
            cfg['handlers']['sentry'] = {
                'level': 'ERROR',
                'class': 'raven.handlers.logging.SentryHandler',
                'dsn': self.cfg.sentry_dsn,
                'tags': {
                    'operator': self.cfg.operator,
                },
            }
            cfg['loggers']['']['handlers'].append('sentry')
        return cfg

    def get_distributed_lock(self):
        return kubiki.distributed_lock.YtLock(**self.cfg.yt_locks)

    def make_stat_monitor(self):
        if self.cfg.solomon:
            statmon = FeedServiceMonitor(self, backend_params=self.cfg.solomon)
        else:
            statmon = DummyServiceMonitor(self)
        return statmon

    def _make_request(self, method, url, headers, json):
        request = requests.Request(method, url, headers=headers, json=json)
        prepared_request = self._session.prepare_request(request)
        return prepared_request

    def _request(self, url, headers=None, method='GET', json=None, timeout=None):
        if timeout is None:
            timeout = self.request_timeout

        req = self._make_request(method=method, url=url, headers=headers, json=json)
        resp = self._session.send(req, timeout=timeout)
        resp.raise_for_status()

        return resp

    def get_cars(self):
        download_start = time.time()

        feed = self._get_feed()

        self.stat_monitor.set_value(None,
                                    time.time() - download_start,
                                    component='upstream',
                                    subcomponent='api',
                                    subsubcomponent='latency')
        self.stat_monitor.set_value(None,
                                    len(feed),
                                    component='upstream',
                                    subcomponent='api',
                                    subsubcomponent='data_volume')

        data = {
            'feed': feed,
            'operator': self.cfg.operator.short_name,
        }
        self._push_client.log(type_='feed', operator=self.cfg.operator.short_name, data=data)

        cars = self._parse_feed(feed, updated_at=download_start)

        return cars

    @classmethod
    def parse_feed(cls, operator, feed, updated_at=None):
        feed_class = cls.registry[operator]
        return feed_class._parse_feed(  # pylint: disable=protected-access
            feed, updated_at=updated_at,
        )

    @classmethod
    def _parse_feed(cls, feed, updated_at=None):
        cars = []
        for item in feed:
            try:
                car = cls.CarClass.from_raw(
                    cls.operator,
                    item,
                    is_free=True,
                    updated_at=updated_at,
                )
            except Exception:
                LOGGER.exception('Failed to parse car data')
                continue
            cars.append(car)
        return cars

    def _get_feed(self):
        resp = self._request(url=self.request_url, headers=self.request_headers)
        feed = self._extract_feed(resp)
        return feed

    def update_index(self, cars):
        upload_start = time.time()
        stats = self._index.update(cars)
        self._push_client.log(type_='saas-stats',
                              operator=self.cfg.operator.short_name,
                              data=stats.stats_dict)
        self.stat_monitor.set_value(None,
                                    time.time() - upload_start,
                                    component='downstream',
                                    subcomponent='saas',
                                    subsubcomponent='latency')
        self.stat_monitor.set_value(None,
                                    len(cars),
                                    component='downstream',
                                    subcomponent='saas',
                                    subsubcomponent='data_volume')

    def tick(self):
        try:
            cars = self.get_cars()
        except Exception:
            LOGGER.exception('Failed to fetch free cars')
            raise

        LOGGER.info('%s free cars', len(cars))

        try:
            self.update_index(cars)
        except Exception:
            LOGGER.exception('Failed to update the index')
            raise
