from __future__ import unicode_literals
import logging

from ..base.car import Car


LOGGER = logging.getLogger(__name__)


class BelkaCar(Car):

    _TRANSMISSION_MAP = {
        'automatic': Car.TRANSMISSION.AUTOMATIC,
    }

    @classmethod
    def extract_local_id(cls, raw_data):
        return raw_data['id']

    @classmethod
    def extract_color(cls, raw_data):
        return None

    @classmethod
    def extract_fuel(cls, raw_data):
        model = cls.extract_model(raw_data)
        if model == 'KIA Rio':
            max_no_refill_distance = 350
        elif model == 'Ford Fiesta':
            max_no_refill_distance = 300
        else:
            # KIA Rio is far more popular so fallback to it.
            max_no_refill_distance = 350

        try:
            fuel = float(raw_data['no_refill_distance']) / max_no_refill_distance
        except Exception:
            fuel = 0
            LOGGER.exception('Failed to estimate fuel level')

        fuel = max(0.0, min(1.0, fuel)) * 100

        return fuel

    @classmethod
    def extract_lat(cls, raw_data):
        return raw_data['latitude']

    @classmethod
    def extract_lon(cls, raw_data):
        return raw_data['longitude']

    @classmethod
    def extract_model(cls, raw_data):
        type_info = raw_data['type']
        brand = type_info['brand']
        model = type_info['model']
        full_model = '{} {}'.format(brand, model)
        return full_model

    @classmethod
    def extract_plate_number(cls, raw_data):
        return raw_data['gos_number']

    @classmethod
    def extract_parking_tariff(cls, raw_data):
        model = cls.extract_model(raw_data)
        if 'mercedes' in model.lower():
            tariff = 4.0
        else:
            tariff = 2.0
        return tariff

    @classmethod
    def extract_usage_tariff(cls, raw_data):
        model = cls.extract_model(raw_data)
        if 'mercedes' in model.lower():
            tariff = 16.0
        else:
            tariff = 8.0
        return tariff

    @classmethod
    def extract_transmission(cls, raw_data):
        return cls._map_transmission(raw_data['type']['transmission_type'])
