from __future__ import unicode_literals
import logging
import os

import furl

from cars.aggregator.static_data import operators
from ..base.config import FeedConfig
from ..base.service import Feed
from .car import BelkaCar


LOGGER = logging.getLogger(__name__)


class BelkaFeedConfig(FeedConfig):

    api_token = os.environ.get('CARSHARING_BELKA_API_TOKEN')
    api_base_url = 'https://api.belkacar.ru'
    api_latitude = 55.733931
    api_longitude = 37.588137
    api_radius = 100000

    @property
    def api_args(self):
        args = {
            'token': self.api_token,
            'device_id': 'yandex_client',
            'latitude': self.api_latitude,
            'longitude': self.api_longitude,
            'filters[radius] ': self.api_radius,
        }
        return args

    @property
    def api_cars_url(self):
        path = '/v1.6/car/'
        url = furl.furl(self.api_base_url).set(path=path, args=self.api_args)
        return url.url

    @property
    def api_car_types_url(self):
        path = '/v1.6/car-types'
        url = furl.furl(self.api_base_url).set(path=path, args=self.api_args)
        return url.url


class BelkaFeed(Feed):

    operator = operators.BELKA
    CarClass = BelkaCar
    ConfigClass = BelkaFeedConfig

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._car_types = {}  # type_id -> type_info

    @property
    def request_url(self):
        return self.cfg.api_cars_url

    @property
    def request_timeout(self):
        return 30

    def startup(self, *args, **kwargs):  # pylint: disable=arguments-differ
        super().startup(*args, **kwargs)
        try:
            response = self._request(self.cfg.api_car_types_url)
            data = response.json()
            self._car_types = {ct['id']: ct for ct in data['car_types']}
        except Exception:
            LOGGER.exception('Failed to fetch car types')
            raise

    def _extract_feed(self, response):
        data = response.json()
        cars = data['list']
        for car in cars:
            car_type = car['type_id']
            car['type'] = self._car_types[car_type]
        return cars
