import datetime
import json
import logging
import re

from django.db.models import Q, Sum, Count
from django.utils import timezone

from cars.core.util import datetime_helper
from cars.core.new_billing import NewBillingClient
from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.settings import FINES as settings

from cars.fines.models import AutocodeFine

from .fine_notifier import FineNotifier


LOGGER = logging.getLogger(__name__)


class MatcherBase(object):
    ADMISSIBLE_TAG_NAMES = [
        'fine_pdd_8_14_2', 'fine_pdd_8_25', 'fine_pdd_12_02_1', 'fine_pdd_12_02_2', 'fine_pdd_12_03_1',
        'fine_pdd_12_03_2', 'fine_pdd_12_05_1', 'fine_pdd_12_05_3_1', 'fine_pdd_12_06', 'fine_pdd_12_07_1',
        'fine_pdd_12_07_3', 'fine_pdd_12_08_1', 'fine_pdd_12_09_2', 'fine_pdd_12_09_3', 'fine_pdd_12_09_4',
        'fine_pdd_12_09_6', 'fine_pdd_12_09_7', 'fine_pdd_12_10_2', 'fine_pdd_12_11_1', 'fine_pdd_12_11_3',
        'fine_pdd_12_12_1', 'fine_pdd_12_12_2', 'fine_pdd_12_12_3', 'fine_pdd_12_13_1', 'fine_pdd_12_13_2',
        'fine_pdd_12_14_1', 'fine_pdd_12_14_1_1', 'fine_pdd_12_14_2', 'fine_pdd_12_14_3', 'fine_pdd_12_15_1',
        'fine_pdd_12_15_1_1', 'fine_pdd_12_15_2', 'fine_pdd_12_15_3', 'fine_pdd_12_15_4', 'fine_pdd_12_15_5',
        'fine_pdd_12_16_1', 'fine_pdd_12_16_2', 'fine_pdd_12_16_3', 'fine_pdd_12_16_3_1', 'fine_pdd_12_16_4',
        'fine_pdd_12_16_5', 'fine_pdd_12_17_1_1', 'fine_pdd_12_17_1_2', 'fine_pdd_12_17_2', 'fine_pdd_12_18',
        'fine_pdd_12_19_1', 'fine_pdd_12_19_2', 'fine_pdd_12_19_5', 'fine_pdd_12_19_6', 'fine_pdd_12_20',
        'fine_pdd_12_23_1', 'fine_pdd_12_23_3', 'fine_pdd_12_25_2', 'fine_pdd_12_26_1', 'fine_pdd_12_26_2',
        'fine_pdd_12_27_1', 'fine_pdd_12_28_2', 'fine_pdd_12_29_1', 'fine_pdd_12_33', 'fine_pdd_12_36_1',
        'fine_pdd_12_37_1', 'fine_pdd_12_37_2', 'fine_pdd_19_03_1',
    ]

    def match(self, article):
        return None


class GenericArticleMatcher(MatcherBase):
    ARTICLE_RE = re.compile('(\d+(?:[ ]*\.[ ]*\d+)+(?:(?:[ ]*[ч|п])(?:[ ]*\.[ ]*\d+)+)?)')

    def match(self, article):
        needles = self.ARTICLE_RE.findall(article)
        if len(needles) == 1:
            article_parts = re.sub('[^\d\.]', '', needles[0]).split('.')

            if len(article_parts) >= 2:
                article_parts[1] = '%02d' % int(article_parts[1])

            tag_name = '_'.join(['fine_pdd'] + article_parts)
            if tag_name in self.ADMISSIBLE_TAG_NAMES:
                return tag_name

        return None


class CityParkingMatcher(MatcherBase):
    def match(self, article):
        if re.search('платн(ой|ую)[ ]+(городск(ой|ую)[ ]+)?парковк(е|у)', article) is not None:
            return 'fine_pdd_8_14_2'
        return None


class LawnParkingMatcher(MatcherBase):
    def match(self, article):
        if re.search('зелены(е|ми)[ ]+насаждени(я|ями)', article) is not None:
            return 'fine_pdd_8_25'
        return None


FINE_TAG_MATCHERS = [
    GenericArticleMatcher(),
    CityParkingMatcher(),
    LawnParkingMatcher(),
]


class FineUserTagProcessor(object):
    def __init__(self, *, saas_client):
        assert isinstance(saas_client, SaasDriveAdminClient)
        self._saas_client = saas_client

    @classmethod
    def from_settings(cls):
        return cls(
            saas_client=SaasDriveAdminClient.from_settings()
        )

    def _determine_tag(self, fine):
        assert isinstance(fine, AutocodeFine)

        for matcher in FINE_TAG_MATCHERS:
            assert isinstance(matcher, MatcherBase)
            tag_name = matcher.match(fine.article_koap)
            if tag_name is not None:
                return tag_name

        return None

    def charge_fine(self, fine):
        assert isinstance(fine, AutocodeFine)

        needle_tag = self._determine_tag(fine)
        if needle_tag is None:
            message = 'cannot determine tag name for fine {}'.format(fine.id)
            LOGGER.error(message)
            raise Exception(message)

        comment = json.dumps({
            'fine_id': str(fine.id),
            'ruling_number': fine.ruling_number,
            'source_type': fine.source_type,
        })

        extra_data = {
            'amount': int(fine.sum_to_pay * 100),
        }

        if fine.car_id is not None:
            extra_data['car_number'] = str(fine.car.number)

        if fine.session_id is not None:
            extra_data['session_id'] = fine.session_id

        tag_id = None
        try:
            tag_id = self._saas_client.add_user_tag(fine.user_id, needle_tag, comment, re_raise=True, **extra_data)
        except Exception:
            message = 'error adding tag {} for fine {}'.format(needle_tag, fine.id)
            LOGGER.exception(message)
            raise

        return tag_id

    def check_fine_charge_tag_exists(self, fine):
        assert isinstance(fine, AutocodeFine)

        tag_id_needle = None
        meta_info = fine.parsed_meta_info
        if meta_info is not None and 'tag_id' in meta_info:
            tag_id_needle = meta_info['tag_id']

        try:
            tags = self._saas_client.get_user_tags(fine.user_id, re_raise=True)
        except Exception:
            LOGGER.exception('cannot get user tags for fine {}'.format(fine.id))
            raise

        for tag in tags:
            if tag_id_needle is not None:
                if tag['tag_id'] == tag_id_needle:
                    return True
            else:
                try:
                    tag_comment_data = json.loads(tag['comment'])
                    if tag_comment_data.get('fine_id', '') == str(fine.id):
                        return True
                except Exception:
                    pass

        return False


class FineCollector(object):
    SMS_PUSH_SWITCH_NOTIFICATION_DATE = datetime_helper.utc_localize(datetime.datetime(2019, 6, 28, 0, 0, 0))
    TAG_CHARGE_SWITCH_DATE = datetime_helper.utc_localize(datetime.datetime(2019, 12, 3, 0, 0, 0))

    def __init__(self, *, billing_client, saas_client, notifier, charge, send_email,
                 send_sms, send_push, yandexoid_only, drive_only, emails_only,
                 charge_limit, charge_time_limit):
        self._fine_user_tag_processor = FineUserTagProcessor(saas_client=saas_client)

        self._billing_client = billing_client
        self._notifier = notifier
        self._charge = charge
        self._send_email = send_email
        self._send_sms = send_sms
        self._send_push = send_push
        self._yandexoid_only = yandexoid_only
        self._drive_only = drive_only
        self._emails_only = emails_only
        self._charge_limit = charge_limit
        self._charge_time_limit = charge_time_limit

    @classmethod
    def from_settings(cls):
        return cls(
            billing_client=NewBillingClient.from_settings(),
            saas_client=SaasDriveAdminClient.from_settings(),
            notifier=FineNotifier.from_settings(),
            charge=settings['charge'],
            send_email=settings['send_email'],
            send_sms=settings['send_sms'],
            send_push=settings['send_push'],
            yandexoid_only=settings['yandexoid_only'],
            drive_only=settings['drive_only'],
            emails_only=settings['emails_only'],
            charge_limit=settings['charge_limit'],
            charge_time_limit=settings['charge_time_limit'],
        )

    def get_collectable_query(
            self, *,
            charge_limit=None, charge_time_limit=None, source_type=None,
            article_koap=None, article_koap_like=None, article_koap_not_like=None,
            ruling_date_since=None, ruling_date_until=None,
            info_received_since=None, info_received_until=None,
            violation_since=None, violation_until=None,
            has_discount=None, is_charged=None, is_charge_passed=None
    ):
        if not self._charge:
            return Q(pk__isnull=True)  # always false
        q = Q(charge_passed_at=None)
        if self._send_push:
            q |= (Q(charge_push_sent_at=None) & Q(charged_at__gte=self.SMS_PUSH_SWITCH_NOTIFICATION_DATE))
        if self._send_sms:
            q |= Q(charge_sms_sent_at=None)
        if self._send_email:
            q |= Q(charge_email_sent_at=None)
        if self._yandexoid_only:
            q &= Q(user__is_yandexoid=True)
        if self._drive_only:
            q &= Q(user__tags__contains='{drive_staff}')
        if self._emails_only is not None:
            q &= Q(user__email__in=self._emails_only)
        charge_limit = charge_limit or self._charge_limit
        if charge_limit:
            q &= Q(sum_to_pay__lte=charge_limit)
        if ruling_date_since is not None or ruling_date_until is not None:
            if ruling_date_since is not None:
                q &= Q(ruling_date__gte=ruling_date_since)
            if ruling_date_until is not None:
                q &= Q(ruling_date__lt=ruling_date_until)
        else:
            charge_time_limit = charge_time_limit or self._charge_time_limit
            if charge_time_limit:
                q &= Q(ruling_date__gte=timezone.now() - charge_time_limit)
        if info_received_since is not None or info_received_until is not None:
            if info_received_since is not None:
                q &= Q(fine_information_received_at__gte=info_received_since)
            if info_received_until is not None:
                q &= Q(fine_information_received_at__lt=info_received_until)
        if violation_since is not None or violation_until is not None:
            if violation_since is not None:
                q &= Q(violation_time__gte=violation_since)
            if violation_until is not None:
                q &= Q(violation_time__lt=violation_until)
        if source_type is not None:
            q &= Q(source_type=source_type)
        if article_koap is not None:
            q &= Q(article_koap=article_koap)
        if article_koap_like is not None:
            q &= Q(article_koap__contains=article_koap_like)
        if article_koap_not_like is not None:
            q &= ~Q(article_koap__contains=article_koap_not_like)
        if has_discount is not None:
            q &= Q(discount_date__isnull=(not has_discount))
        if is_charged is not None:
            q &= Q(charged_at__isnull=(not is_charged))
        if is_charge_passed is not None:
            q &= Q(charge_passed_at__isnull=(not is_charge_passed))
        q &= Q(needs_charge=True)
        return q

    def collect(self, fine, email_campaign_name=None, sum_limit_per_day=None, count_limit_per_day=None):
        assert isinstance(fine, AutocodeFine)

        updated = False

        photo_receive_timeout_seconds = 3600
        if (fine.has_photo and not fine.photos.count()  # lack of photo
            and (fine.fine_information_received_at >  # give time to load the photos
                 timezone.now() - datetime.timedelta(seconds=photo_receive_timeout_seconds))):
            return updated

        if fine.charged_at is None:
            if sum_limit_per_day is not None or count_limit_per_day is not None:
                qs = AutocodeFine.objects.filter(
                    user_id=fine.user_id,
                    charged_at__gte=datetime.date.today()
                ).aggregate(amount_paid=Sum('sum_to_pay'), total_fines=Count('user_id'))

                amount_paid = qs.get('amount_paid', 0) or 0
                total_fines = qs.get('total_fines', 0) or 0

                if amount_paid >= sum_limit_per_day or total_fines >= count_limit_per_day:
                    LOGGER.info(
                        'stop processing fine #{} by day limit: charged amount - {} of {}, count - {} of {}'
                        .format(fine.ruling_number, amount_paid, sum_limit_per_day, total_fines, count_limit_per_day)
                    )
                    return updated

            charged = self._charge_fine(fine)
            updated = updated or charged

        if not fine.charged_at:
            return updated  # if failed to charge money - do nothing

        if fine.charge_passed_at is None:
            charge_passed = self.check_fine_charged(fine)
            updated = updated or charge_passed

        # if not fine.charge_passed_at:
        #     return  # if money are not our yet - don't send notifications

        self._notify(fine, email_campaign_name)
        return updated

    def _charge_fine(self, fine):
        assert isinstance(fine, AutocodeFine)
        tag_id = None
        try:
            if not self._check_charge_exists(fine):
                tag_id = self._fine_user_tag_processor.charge_fine(fine)
        except Exception:
            LOGGER.exception('Failed to sent request to charge money for fine {}'.format(fine.ruling_number))
            return False

        if tag_id is not None:
            fine.update_meta_info(tag_id=tag_id)
        fine.charged_at = timezone.now()
        fine.save()
        LOGGER.info('Successfully sent request to charge money for fine {}'.format(fine.ruling_number))
        return True

    def _check_charge_exists(self, fine):
        return (self._billing_client.charge_exists(session_id=fine.id, user_id=fine.user_id) or
                self._fine_user_tag_processor.check_fine_charge_tag_exists(fine))

    def check_fine_charged(self, fine):
        try:
            if self._check_charge_has_passed(fine):
                fine.charge_passed_at = timezone.now()
                fine.save()
        except Exception:
            LOGGER.exception('Failed to get charge status for fine {}'.format(fine.ruling_number))
            return False

        if fine.charge_passed_at is not None:
            LOGGER.info('Successfully updated charge status for fine {}'.format(fine.ruling_number))
            return True

        return False

    def _check_charge_has_passed(self, fine):
        if fine.charged_at < self.TAG_CHARGE_SWITCH_DATE:
            return self._billing_client.charge_is_passed(session_id=fine.id, user_id=fine.user_id)

        return not self._fine_user_tag_processor.check_fine_charge_tag_exists(fine)

    def _notify(self, fine, email_campaign_name=None):
        if (
                self._send_push and
                fine.charge_push_sent_at is None and
                fine.charged_at >= self.SMS_PUSH_SWITCH_NOTIFICATION_DATE
        ):
            try:
                self._notifier.push_notify_fine_charged(fine)
            except Exception:
                LOGGER.exception('Failed to send push about fine charge')
            else:
                fine.charge_push_sent_at = timezone.now()
                fine.save()
                LOGGER.info('Successfully sent push to user {}'.format(fine.user_id))

        if self._send_sms and fine.charge_sms_sent_at is None:
            try:
                if fine.user.phone is None:
                    raise Exception('user with id {} has no phone number'.format(fine.user_id))

                self._notifier.sms_notify_fine_charged(fine)
            except Exception:
                LOGGER.exception('Failed to send sms about fine charge to user {}'.format(fine.user_id))
            else:
                fine.charge_sms_sent_at = timezone.now()
                fine.save()
                LOGGER.info('Successfully sent sms to user {}'.format(fine.user_id))

        if self._send_email and fine.charge_email_sent_at is None:
            try:
                self._notifier.email_notify_fine_charged(fine, campaign_name=email_campaign_name)
            except Exception:
                LOGGER.exception('Failed to send email fine charge')
            else:
                fine.charge_email_sent_at = timezone.now()
                fine.save()
                LOGGER.info('Successfully sent email to user {}'.format(fine.user_id))
