import json
import logging

import cars.settings

from django.db.models import Max
from cars.carsharing.models.tag import CarTag

from cars.request_aggregator.core.common_helper import YTHelper
from cars.request_aggregator.core.request_time_sync_helper import RequestEntryCountSyncHelper

from cars.request_aggregator.models.call_center_common import SyncOrigin

from cars.fines.models.fine import AutocodeFine


LOGGER = logging.getLogger(__name__)


class FinesExportSyncingHelper(RequestEntryCountSyncHelper):
    @classmethod
    def from_settings(cls):
        return cls(stat_sync_origin=SyncOrigin.FINES_EXPORTING_EXT)


class FinesExportHelper(object):
    CHUNKS_MERGE_LIMIT = 500

    DEFAULT_CITY = 'MSK'

    CITY_TAGS = {
        'SPB': 'peterburg_tag',
        'KZN': 'kazan_tag',
    }

    def __init__(self, yt_export_table):
        self._yt_helper = YTHelper(yt_export_table)
        self._sync_helper = FinesExportSyncingHelper.from_settings()

        self._car_sources = None

    @classmethod
    def from_settings(cls):
        return cls(cars.settings.FINES['export_table'])

    def init_car_sources(self):
        car_sources = {}

        for city, city_tag in self.CITY_TAGS.items():
            car_ids = (
                CarTag.objects.using(cars.settings.DB_RO_ID)
                .filter(tag=city_tag)
                .values_list('object_id', flat=True)
            )

            for car_id in car_ids:
                car_sources[str(car_id)] = city

        self._car_sources = car_sources

    def export(self):
        if self._car_sources is None:
            self.init_car_sources()

        schema = self._get_schema()

        max_serial_id = AutocodeFine.objects.aggregate(Max('serial_id'))['serial_id__max']

        for idx, (start_serial_id, end_serial_id) in enumerate(self._sync_helper.iter_indexes_to_process(max_serial_id)):
            data = self._iter_data_to_store(start_serial_id, end_serial_id)
            self._yt_helper.export_data(data, schema=schema)

            if not (idx + 1) % self.CHUNKS_MERGE_LIMIT:
                self.optimize()

    def _get_schema(self):
        return [
            {'name': 'id', 'type': 'string', 'required': True},
            {'name': 'serial_id', 'type': 'int64', 'required': True},
            {'name': 'ruling_number', 'type': 'string', 'required': True},
            {'name': 'ruling_date', 'type': 'string'},
            {'name': 'violation_time', 'type': 'double'},
            {'name': 'violation_place', 'type': 'string'},
            {'name': 'sum_to_pay_without_discount', 'type': 'double'},
            {'name': 'sum_to_pay', 'type': 'double'},
            {'name': 'discount_date', 'type': 'string'},
            {'name': 'article_koap', 'type': 'string'},
            {'name': 'violation_document_number', 'type': 'string'},
            {'name': 'violation_document_type', 'type': 'string'},
            {'name': 'autocode_id', 'type': 'int64'},
            {'name': 'autocode_payment_confirmation_id', 'type': 'int64'},
            {'name': 'order_id', 'type': 'string'},
            {'name': 'user_id', 'type': 'string'},
            {'name': 'session_id', 'type': 'string'},
            {'name': 'car_id', 'type': 'string'},
            {'name': 'is_camera_fixation', 'type': 'boolean'},
            {'name': 'needs_charge', 'type': 'boolean'},
            {'name': 'has_photo', 'type': 'boolean'},
            {'name': 'violation_latitude', 'type': 'double'},
            {'name': 'violation_longitude', 'type': 'double'},
            {'name': 'is_speed_violation', 'type': 'boolean'},
            {'name': 'is_after_ride_start_during_order', 'type': 'boolean'},
            {'name': 'added_at_timestamp', 'type': 'int64'},
            {'name': 'fine_information_received_at', 'type': 'double'},
            {'name': 'payment_confirmation_received_at', 'type': 'double'},
            {'name': 'charged_at', 'type': 'double'},
            {'name': 'charge_passed_at', 'type': 'double'},
            {'name': 'charge_email_sent_at', 'type': 'double'},
            {'name': 'charge_sms_sent_at', 'type': 'double'},
            {'name': 'charge_push_sent_at', 'type': 'double'},
            {'name': 'odps_code', 'type': 'string'},
            {'name': 'odps_name', 'type': 'string'},
            {'name': 'skipped', 'type': 'int64'},
            {'name': 'meta_info', 'type': 'string'},
            {'name': 'source', 'type': 'string'},
            {'name': 'source_type', 'type': 'string'},
            {'name': 'city', 'type': 'string'},
            {'name': 'charge_tag_id', 'type': 'string'},
        ]

    def _iter_data_to_store(self, start_serial_id, end_serial_id):
        fine = None

        fine_ids = list(
            AutocodeFine.objects.using(cars.settings.DB_RO_ID)
            .filter(serial_id__gte=start_serial_id, serial_id__lt=end_serial_id)
            .values_list('id', flat=True)
        )

        chunk_size = 50

        try:
            for chunk_offset in range(0, len(fine_ids), chunk_size):
                fine_ids_chunk = fine_ids[chunk_offset:chunk_offset + chunk_size]
                fines_chunk = AutocodeFine.objects.using(cars.settings.DB_RO_ID).filter(id__in=fine_ids_chunk)
                for fine in fines_chunk:
                    yield self._process_fine(fine)
        except Exception:
            fine_id = getattr(fine, 'id', None)
            LOGGER.exception('error processing fine with id {}'.format(fine_id))
            raise

    def _process_fine(self, fine):
        assert isinstance(fine, AutocodeFine)

        formatted_fine = {}

        formatted_fine['id'] = str(fine.id)
        formatted_fine['serial_id'] = fine.serial_id

        formatted_fine['ruling_number'] = fine.ruling_number
        formatted_fine['ruling_date'] = fine.ruling_date.strftime('%Y-%m-%d')

        formatted_fine['violation_time'] = fine.violation_time.timestamp() if fine.violation_time else None
        formatted_fine['violation_place'] = fine.violation_place

        formatted_fine['sum_to_pay_without_discount'] = float(fine.sum_to_pay_without_discount)
        formatted_fine['sum_to_pay'] = float(fine.sum_to_pay)
        formatted_fine['discount_date'] = fine.discount_date.strftime('%Y-%m-%d') if fine.discount_date else None

        formatted_fine['article_koap'] = fine.article_koap

        formatted_fine['violation_document_number'] = fine.violation_document_number
        formatted_fine['violation_document_type'] = fine.violation_document_type

        formatted_fine['autocode_id'] = fine.autocode_id
        formatted_fine['autocode_payment_confirmation_id'] = fine.autocode_payment_confirmation_id

        formatted_fine['order_id'] = str(fine.order_id) if fine.order_id else None
        formatted_fine['user_id'] = str(fine.user_id) if fine.user_id else None
        formatted_fine['session_id'] = str(fine.session_id) if fine.session_id else None
        formatted_fine['car_id'] = str(fine.car_id) if fine.car_id else None

        formatted_fine['is_camera_fixation'] = fine.is_camera_fixation
        formatted_fine['needs_charge'] = fine.needs_charge

        formatted_fine['has_photo'] = fine.has_photo

        formatted_fine['violation_latitude'] = fine.violation_latitude
        formatted_fine['violation_longitude'] = fine.violation_longitude

        formatted_fine['is_speed_violation'] = self._check_is_speed_violation(fine)
        formatted_fine['is_after_ride_start_during_order'] = fine.is_after_ride_start_during_order

        formatted_fine['added_at_timestamp'] = fine.added_at_timestamp
        formatted_fine['fine_information_received_at'] = fine.fine_information_received_at.timestamp()
        formatted_fine['payment_confirmation_received_at'] = fine.payment_confirmation_received_at.timestamp() if fine.payment_confirmation_received_at else None

        formatted_fine['charged_at'] = fine.charged_at.timestamp() if fine.charged_at else None
        formatted_fine['charge_passed_at'] = fine.charge_passed_at.timestamp() if fine.charge_passed_at else None
        formatted_fine['charge_email_sent_at'] = fine.charge_email_sent_at.timestamp() if fine.charge_email_sent_at else None
        formatted_fine['charge_sms_sent_at'] = fine.charge_sms_sent_at.timestamp() if fine.charge_sms_sent_at else None
        formatted_fine['charge_push_sent_at'] = fine.charge_push_sent_at.timestamp() if fine.charge_push_sent_at else None

        formatted_fine['odps_code'] = fine.odps_code
        formatted_fine['odps_name'] = fine.odps_name

        formatted_fine['skipped'] = fine.skipped

        formatted_fine['source'] = getattr(fine, 'source_type', None) or 'autocode'
        formatted_fine['source_type'] = formatted_fine['source']
        formatted_fine['city'] = self._car_sources.get(str(fine.car_id), self.DEFAULT_CITY)

        formatted_fine['meta_info'] = fine.meta_info if fine.meta_info else '{}'

        meta_info = json.loads(fine.meta_info) if fine.meta_info else {}
        formatted_fine['charge_tag_id'] = meta_info.get('tag_id', None)

        return formatted_fine

    def _check_is_speed_violation(self, fine):
        return fine.article_koap.lstrip().startswith(('12.9.', '12.09.', 'КоАП 12.9'))

    def optimize(self):
        self._yt_helper.merge_chunks()
